#! perl
#
###########################################################
# kmm.pl version 0.3 
# Copyright (C) 2004 Chuck Chargin Jr. (cchargin@comcast.net)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
##########################################################
# History:
#	
# July 10 2004: First public release of kmm.pl version 0.1
#                
# July 20 2004: version 0.2
#               Added .uts to mod extensions (thanks Darth333)
#               Changed 'system del' to unlink (thanks tk102)
#               Changed list boxes to trees
#               Added mod groups
#               Added 'deactivate all' button
#
# August 25 2004: version 0.2a
#                 Added .lip to mod extensions (thanks RedHawke)
#                 Fixed a bug that could leave a file destination empty
#
# October 18 2004: version 0.2b
#                 Added .jrl to mod extensions (thanks RedHawke)
#                 Added .rim to mod extensions (thanks Shangoman)
#
# February 14 2005: version 0.3
#                   Now works with Kotor 2!  All your mods in 1 place!
#                   Now works with Windows 98.  Yes, Windows 98 (thanks Kevin)
#               
##########################################################
#
# This is a script to manage kotor mods.
#
# features:
# -keeps track of all your installed mods for Kotor 1 and 2
# -keeps your kotor 1 and kotor 2 mods separate
# -when a mod is activated it copies the files to override
# -when a mod is de-activated it removes the files from override
# -supports mods that put files in places other than override
# -checks for conflicts between mods
# -mod groups let you activate a bunch of mods with 1 click
#
# usage:
# perl script: perl perl.pl
#
# compiled script: double click kmm.exe
#
# see readme for more info and ditribution information

use strict;
use Tk;
use Tk::Tree;
use File::Find;
use Win32::FileOp;
use Win32::GUI;
use Win32::TieRegistry (Delimiter => "/");
use Cwd;

use vars qw($VERSION);
$VERSION = '0.3';

#read the location of kotor from the registry
our %kotordir;
$kotordir{'kotor1'} = $Registry->{"LMachine/Software/bioware/sw/kotor/path"} . "\\";
$kotordir{'kotor2'} = $Registry->{"LMachine/Software/LucasArts/KotOR2/Path"} . "\\";

#our @activemods;     # all of the currently active mods
#our @inactivemods;   # all of the currently inactive mods
#our %modslist;       # all the mods and their info
#our $moddir;         # where the user says we should find the mods
#our %modlookup;      # has mod names as keys and the index as values
#our @modindex;       # index corresponds to the value in %modlookup, value is the mod name
#our %modgroups;      # holds the mod groups

# the "big hash", atm = AllTheMods
our %atm = ( 'kotor1' => { 'activemods' => [],
		           'inactivemods' => [],
			   'modslist' => {},
			   'moddir' => "",
			   'modlookup' => {},
			   'modindex' => [],
			   'modgroups' => {}
			 },
             'kotor2' => { 'activemods' => [],
		           'inactivemods' => [],
			   'modslist' => {},
			   'moddir' => "",
			   'modlookup' => {},
			   'modindex' => [],
			   'modgroups' => {}
			 }
           );


our $workingdir;     # where KMM is running from
our $mode = 'kotor1';    # 'kotor' for kotor 1, 'kotor2' for kotor2

our $texteditor = "notepad";  # the default text editor

# if the file extension is not in this list then kmm will not touch it 
our %modextensions = ('.uti' => 1, '.tga' => 1, '.2da' => 1, '.mdl' => 1, '.mdx' => 1, '.mod' => 1, '.utc' => 1, '.utp' => 1, '.txi' => 1, '.lyt' => 1, '.dlg' => 1, '.ncs' => 1, '.wav' => 1, '.utm' => 1, '.itp' => 1, '.uts' => 1, '.lip' => 1, '.rim' => 1, '.jrl' => 1);

our %object;
$object{'main'} = MainWindow->new;
$object{'main'}->configure(-title => "Kotor Mod Manager (ver: $VERSION)");

our ($maxwidth, $maxheight) = $object{'main'}->maxsize;

$object{'inactivelist'} = $object{'main'}->Scrolled('Tree',-itemtype => 'text',
                                                     -browsecmd => \&inactiveclick,
						     -width => 55,
						     -height => 18,
						     -selectbackground => 'darkblue',
						     -selectforeground => 'white',
                                                     -scrollbars => 'e');

$object{'activelist'} = $object{'main'}->Scrolled('Tree',-itemtype => 'text',
                                                     -browsecmd => \&activeclick,
						     -width => 55,
						     -height => 18,
						     -selectbackground => 'darkblue',
						     -selectforeground => 'white',
                                                     -scrollbars => 'e');
					     
$object{'inactivetext'} = $object{'main'}->Label(-text => 'Inactive mods:');
$object{'activetext'} = $object{'main'}->Label(-text => 'Active mods:');
$object{'kotor1cb'} = $object{'main'}->Radiobutton(-text => "Kotor 1", -indicatoron => 'false',
	                                           -variable => \$mode, -value => 'kotor1',
					           -command => \&changemode);
$object{'kotor2cb'} = $object{'main'}->Radiobutton(-text => "Kotor 2", -indicatoron => 'false',
	                                           -variable => \$mode, -value => 'kotor2',
					           -command => \&changemode);
$object{'addmodbutton'} = $object{'main'}->Button(-text=>"Add mod", -command => \&addmod);
$object{'delmodbutton'} = $object{'main'}->Button(-text=>"Delete mod", -command => \&deletemod);
$object{'activatemodbutton'} = $object{'main'}->Button(-text=>"Activate mod", -command => \&activatemod);
$object{'deactivatemodbutton'} = $object{'main'}->Button(-text=>"Deactivate mod", -command => \&deactivatemod);
$object{'deactivateallbutton'} = $object{'main'}->Button(-text=>"Deactivate ALL", -command => [\&deactivatemod, "all"]);
$object{'describebutton'} = $object{'main'}->Button(-text=>"Mod description", -command => \&describemod);
$object{'creategroupbutton'} = $object{'main'}->Button(-text=>"Create group", -command => \&creategroup);

$object{'inactivetext'}->form(-t => '%0',
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'kotor1cb'}->form(-t => '%0',
	                  -tp => 10,
			  -l => [$object{'inactivetext'},0],
			  -lp => 80);

$object{'kotor2cb'}->form(-t => '%0',
	                  -tp => 10,
			  -l => [$object{'kotor1cb'},0],
			  -lp => 5);

$object{'inactivelist'}->form(-t => [$object{'inactivetext'},0],
	                 -tp => 10,
			 -l => '%0',
			 -lp => 10,
			 -b => [$object{'activetext'},0],
			 -bp => 10,
		         -r => [$object{'describebutton'},0]);

$object{'activetext'}->form(-t => '%45',
	                    -tp => 10,
			    -l => '%0',
			    -lp => 10);
		      
$object{'activelist'}->form(-t => [$object{'activetext'},0],
	                 -tp => 10,
			 -l => '%0',
			 -lp => 10,
			 -b => '%100',
			 -bp => 10,
		         -r => [$object{'describebutton'},0]);

$object{'activatemodbutton'}->form(-l => [$object{'inactivelist'},0],
	                           -lp => 10,
			           -t => '%0',
			           -tp => 10);

$object{'deactivatemodbutton'}->form(-l => [$object{'inactivelist'},0],
	                             -lp => 10,
			             -t => [$object{'activatemodbutton'},0],
			             -tp => 10);

$object{'deactivateallbutton'}->form(-l => [$object{'inactivelist'},0],
	                             -lp => 10,
			             -t => '%25',
			             -tp => 10);

$object{'describebutton'}->form(-r => '%100',
	                        -lp => 10,
			        -t => '%45',
			        -rp => 10);

$object{'creategroupbutton'}->form(-r => '%100',
	                           -rp => 10,
				   -t => [$object{'describebutton'},0],
				   -tp => 10);

$object{'addmodbutton'}->form(-l => [$object{'activelist'},0],
	                      -lp => 10,
		              -b => [$object{'delmodbutton'},0],
		              -bp => 10);

$object{'delmodbutton'}->form(-l => [$object{'activelist'},0],
	                      -lp => 10,
		              -b => '%100',
		              -bp => 10);

$object{'main'}->update;
$object{'main'}->geometry('+' . int($maxwidth/2 - $object{'main'}->width/2) . '+' . int($maxheight/2 - $object{'main'}->height/2) ); 

# create set prefs window
$object{'setprefs'} = $object{'main'}->Toplevel('title' => 'KMM: Set preferences');
$object{'kotordirentry'} = $object{'setprefs'}->Entry(-width => 45);
$object{'kotor2direntry'} = $object{'setprefs'}->Entry(-width => 45);
$object{'moddirentry'} = $object{'setprefs'}->Entry(-width => 45);
$object{'mod2direntry'} = $object{'setprefs'}->Entry(-width => 45);
$object{'kotordirbutton'} = $object{'setprefs'}->Button(-text=>"Kotor directory:", 
	                                                -command => [\&getfolder, 'kotordirentry', ""]);
$object{'kotor2dirbutton'} = $object{'setprefs'}->Button(-text=>"Kotor 2 directory:", 
	                                                -command => [\&getfolder, 'kotor2direntry', ""]);
$object{'moddirbutton'} = $object{'setprefs'}->Button(-text=>"Kotor 1 mod dir:", 
	                                              -command => [\&getfolder, 'moddirentry', ""]);
$object{'mod2dirbutton'} = $object{'setprefs'}->Button(-text=>"Kotor 2 mod dir:", 
	                                              -command => [\&getfolder, 'mod2direntry', ""]);
$object{'setprefsbutton'} = $object{'setprefs'}->Button(-text=>"OK", -command => \&setprefs);
$object{'setprefs'}->protocol('WM_DELETE_WINDOW', [\&withdrawwindow, 'setprefs']);
$object{'setprefs'}->withdraw;

$object{'kotordirbutton'}->form(-t => '%0',
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'kotordirentry'}->form(-t => '%0',
	                       -tp => 10,
			       -l => [$object{'kotordirbutton'},0],
			       -lp => 10,
			       -r => '%100',
			       -rp => 10);
		       
$object{'moddirbutton'}->form(-t => [$object{'kotordirbutton'},0],
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'moddirentry'}->form(-t => [$object{'kotordirentry'},0],
	                       -tp => 15,
			       -l => [$object{'moddirbutton'},0],
			       -lp => 10,
			       -r => '%100',
			       -rp => 10);
		       
$object{'kotor2dirbutton'}->form(-t => [$object{'moddirbutton'},0],
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'kotor2direntry'}->form(-t => [$object{'moddirentry'},0],
	                       -tp => 15,
			       -l => [$object{'kotor2dirbutton'},0],
			       -lp => 10,
			       -r => '%100',
			       -rp => 10);
		       
$object{'mod2dirbutton'}->form(-t => [$object{'kotor2dirbutton'},0],
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'mod2direntry'}->form(-t => [$object{'kotor2direntry'},0],
	                       -tp => 15,
			       -l => [$object{'mod2dirbutton'},0],
			       -lp => 10,
			       -r => '%100',
			       -rp => 10);
		       
$object{'setprefsbutton'}->form(-l => '%45',
	                        -t => [$object{'mod2dirbutton'},0],
				-tp => 10);

$object{'setprefs'}->update;
$object{'setprefs'}->geometry('+' . int($maxwidth/2 - $object{'setprefs'}->width/2) . '+' . int($maxheight/2 - $object{'setprefs'}->height/2) ); 

# create data entry window
$object{'dataentrywin'} = $object{'main'}->Toplevel('title' => 'KMM: Data entry');
$object{'datatext'} = $object{'dataentrywin'}->Label(-text => "Group name:", -width => 12);
$object{'dataentry'} = $object{'dataentrywin'}->Entry(-width => 45);
$object{'commitdatabutton'} = $object{'dataentrywin'}->Button(-text=>"OK", -command => \&commitdata);
$object{'dataentrywin'}->protocol('WM_DELETE_WINDOW', [\&withdrawwindow, 'dataentrywin']);
$object{'dataentrywin'}->withdraw;

$object{'datatext'}->form(-l => '%0',
	                  -lp => 10,
			  -t => '%0',
			  -tp => 10);

$object{'dataentry'}->form(-l => [$object{'datatext'},0],
	                   -lp => 10,
			   -t => '%0',
			   -tp => 10,
			   -r => '%100',
			   -rp => 10);

$object{'commitdatabutton'}->form(-l => '%45',
	                          -t => [$object{'dataentry'},0],
				  -tp => 10);

$object{'dataentrywin'}->update;
$object{'dataentrywin'}->geometry('+' . int($maxwidth/2 - $object{'dataentrywin'}->width/2) . '+' . int($maxheight/2 - $object{'dataentrywin'}->height/2) ); 
			
# create add mod window
$object{'addmodwin'} = $object{'main'}->Toplevel('title' => 'Add mod');
$object{'addmodwin'}->protocol('WM_DELETE_WINDOW', [\&withdrawwindow, 'addmodwin']);
$object{'addmodwin'}->withdraw;
$object{'modnametext'} = $object{'addmodwin'}->Label(-text => "Mod name:", -width => 12);
$object{'modnameentry'} = $object{'addmodwin'}->Entry();
$object{'modauthortext'} = $object{'addmodwin'}->Label(-text => "Author:", -width => 12);
$object{'modauthorentry'} = $object{'addmodwin'}->Entry();
$object{'modvertext'} = $object{'addmodwin'}->Label(-text => "Mod version:", -width => 12);
$object{'modverentry'} = $object{'addmodwin'}->Entry();
$object{'moddescfiletext'} = $object{'addmodwin'}->Label(-text => "Description file:", -width => 12);
$object{'moddescfileentry'} = $object{'addmodwin'}->Entry();
$object{'modpathbutton'} = $object{'addmodwin'}->Button(-text=>"Choose dir", -command => \&addmod);
$object{'modpathentry'} = $object{'addmodwin'}->Entry(-width => 55);
$object{'hiddenentry'} = $object{'addmodwin'}->Entry();
$object{'filelist'} = $object{'addmodwin'}->Scrolled('Listbox', 
                                             -scrollbars => 'se',
			                     -width => 75,
			                     -height => 12);
$object{'commitbutton'} = $object{'addmodwin'}->Button(-text=>"Add the mod", -command => [\&commitaddmod, 'add']);
$object{'modsupbutton'} = $object{'addmodwin'}->Button(-text=>"MOD/SUP", -command => \&modsuptoggle);
$object{'updatekmmbutton'} = $object{'addmodwin'}->Button(-text=>"Add and update kmm", -command => [\&commitaddmod, 'update']);
$object{'destinationbutton'} = $object{'addmodwin'}->Button(-text=>"Destination", -command => \&destination);

$object{'modsupbutton'}-> form(-t => '%35',
	                       -r => '%100',
			       -rp => 10);

$object{'destinationbutton'}->form(-t => [$object{'modsupbutton'},0],
	                           -tp => 10,
				   -r => '%100',
				   -rp => 10);
		       
$object{'modnametext'}->form(-t => '%0',
	                     -tp => 10,
			     -l => '%0',
			     -lp => 10);

$object{'modnameentry'}->form(-t => '%0',
	                      -tp => 10,
			      -l => [$object{'modnametext'},0],
			      -lp => 10,
  		               -r => [$object{'modsupbutton'},0],
		               -rp => 10);

$object{'modauthortext'}->form(-t => [$object{'modnametext'},0],
	                     -tp => 10,
			     -l => '%0',
			     -lp => 10);

$object{'modauthorentry'}->form(-t => [$object{'modnameentry'},0],
	                      -tp => 10,
			      -l => [$object{'modauthortext'},0],
			      -lp => 10,
  		               -r => [$object{'modsupbutton'},0],
		               -rp => 10);

$object{'modvertext'}->form(-t => [$object{'modauthortext'},0],
	                     -tp => 10,
			     -l => '%0',
			     -lp => 10);

$object{'modverentry'}->form(-t => [$object{'modauthorentry'},0],
	                      -tp => 10,
			      -l => [$object{'modvertext'},0],
			      -lp => 10,
  		               -r => [$object{'modsupbutton'},0],
		               -rp => 10);

$object{'moddescfiletext'}->form(-t => [$object{'modvertext'},0],
	                       -tp => 10,
			       -l => '%0',
			       -lp => 10);

$object{'moddescfileentry'}->form(-t => [$object{'modverentry'},0],
	                      -tp => 10,
			      -l => [$object{'moddescfiletext'},0],
			      -lp => 10,
		              -r => [$object{'modsupbutton'},0],
		              -rp => 10);
		      
$object{'modpathbutton'}->form(-t => [$object{'moddescfiletext'},0],
	                       -tp => 10,
			       -l => '%0',
			       -lp => 10,
		               -bp => 10);

$object{'modpathentry'}->form(-t => [$object{'moddescfileentry'},0],
	                      -tp => 10,
			      -l => [$object{'modpathbutton'},0],
			      -lp => 10,
		              -r => [$object{'modsupbutton'},0],
		              -rp => 10);

$object{'filelist'}->form(-t => [$object{'modpathentry'},0],
	                   -tp => 10,
			   -l => '%0',
			   -lp => 10,
			   -r => [$object{'modsupbutton'},0],
			   -rp => 10,
			   -b => [$object{'commitbutton'},0],
			   -bp => 10);
			   
$object{'commitbutton'}->form(-l => '%35',
	                      -b => '%100',
			      -bp => 10,
			      -tp => 10);
		      
$object{'updatekmmbutton'}->form(-l => 	[$object{'commitbutton'},0],
                                 -lp => 10,
			         -b => '%100',
			         -bp => 10);

$object{'addmodwin'}->update;
$object{'addmodwin'}->geometry('+' . int($maxwidth/2 - $object{'addmodwin'}->width/2) . '+' . int($maxheight/2 - $object{'addmodwin'}->height/2) ); 

# initialize the stuff
&kmminit;

# start the freakin' program already!
MainLoop;

sub kmminit {
  my $temp;
  my %grouplookup;
  my $argh;
  my $prefmode;

  $workingdir = getcwd;
  $workingdir =~ s#/#\\#g;
  # load the configuration file or make one
  if (-e $workingdir . "\\kmm.conf") {
    open(KMMCONFIN, $workingdir . "\\kmm.conf") or die "can't open kmm.conf\n";
    while (<KMMCONFIN>) {
      if(/<moddir>=(.*)/) {
        $atm{'kotor1'}{'moddir'} = $1;
      } elsif (/<mod2dir>=(.*)/) {
        $atm{'kotor2'}{'moddir'} = $1;
      } elsif (/<kotordir>=(.*)/) {
        $kotordir{'kotor1'} = $1;
      } elsif (/<kotor2dir>=(.*)/) {
        $kotordir{'kotor2'} = $1;
      } elsif (/<mode>=(.*)/) {
        $prefmode = $1;
      }
    }
    close KMMCONFIN;
  } else { # no configiration file found, so open up a dialog and get the info from the user
    $object{'kotordirentry'}->delete('0', 'end');
    $object{'kotordirentry'}->insert('end', $kotordir{'kotor1'});
  
    $object{'kotor2direntry'}->delete('0', 'end');
    $object{'kotor2direntry'}->insert('end', $kotordir{'kotor2'});

    $object{'moddirentry'}->delete('0', 'end');
    $object{'moddirentry'}->insert('end', $workingdir);

    $object{'mod2direntry'}->delete('0', 'end');
    $object{'mod2direntry'}->insert('end', $workingdir);

    $object{'main'}->iconify;
    $object{'setprefs'}->deiconify;
    $object{'setprefs'}->raise;
    $object{'setprefs'}->focus;
  }

  # if no mode was loaded, or this is a new config, set the mode
  if ($prefmode eq "") {
    if ($kotordir{'kotor1'} ne "") {
      $prefmode = 'kotor1';
    } elsif ($kotordir{'kotor2'} ne "") {
      $prefmode = 'kotor2';
    }
  }

  # create the tree roots
  $object{'inactivelist'}->add('.', 
	                     -text => "Inactive mods",
                             -data => "inactive");

  $object{'activelist'}->add('.', 
	                   -text => "Active mods",
                           -data => "active");

  # load the mods lists for kotor
  foreach $mode ( qw/ kotor1 kotor2 / ) {
    if ($mode eq "kotor1") {
      $argh = "installedmods.kmm";
    } else {
      $argh = "installedmods2.kmm";
    }
    if (-e $workingdir . "\\" . $argh) {
      open(IMKMMIN, $workingdir . "\\" . $argh) or die "can't open $argh\n";
      while (<IMKMMIN>) {
        if (/(.*\\)(.*\....),(.*)/) {
          if (-e $1.$2) {
            &readkmm($1 . $2, $1, 'init');
            $temp = $#{$atm{$mode}{'modindex'}};
            $atm{$mode}{'modslist'}{$temp}{'status'} = $3;
          }
        }
      }
      close IMKMMIN;
    }
  } # foreach ( qw/kotor1 kotor2/ ) {

  # load the groups list for kotor 1
  foreach $mode ( qw/ kotor1 kotor2 / ) {
    if ($mode eq "kotor1") {
      $argh = "modgroups.kmm";
    } else {
      $argh = "modgroups2.kmm";
    }
    if (-e $workingdir . "\\" . $argh) {
      open(kmmgroupin, $workingdir . "\\" . $argh) or die "can't open $argh\n";
      $temp = -1;
      while (<kmmgroupin>) {
        if (/<group>=(.*)/) {
          $temp++;
 	  $grouplookup{$1} = $temp;
          $atm{$mode}{'modgroups'}{$temp} = {};
          $atm{$mode}{'modgroups'}{$temp}{'name'} = $1;
          $atm{$mode}{'modgroups'}{$temp}{'status'} = 'inactive';
          $atm{$mode}{'modgroups'}{$temp}{'mods'} = {};
        } elsif (/<mod>=(.*)/) {
          $atm{$mode}{'modgroups'}{$temp}{'mods'}{$atm{$mode}{'modlookup'}{$1}} = $1;
        }
      }
      close kmmgroupin;
    }
  } # foreach ( qw/kotor1 kotor2/ ) {
  
  # figure out what is active for kotor
  foreach $mode ( qw/ kotor1 kotor2 / ) {
    foreach ( keys %{$atm{$mode}{'modslist'}} ) {
      if ($atm{$mode}{'modslist'}{$_}{'status'} eq "active") {
        push @{$atm{$mode}{'activemods'}}, $_;
      } elsif ($atm{$mode}{'modslist'}{$_}{'status'} eq "inactive" ) {
        push @{$atm{$mode}{'inactivemods'}}, $_;
      } else {
        $atm{$mode}{'modgroups'}{$grouplookup{$atm{$mode}{'modslist'}{$_}{'status'}}}{'status'} = 'active';
      }
    }
    foreach (keys %{$atm{$mode}{'modgroups'}}) {
      if ($atm{$mode}{'modgroups'}{$_}{'status'} eq 'active') {
        push @{$atm{$mode}{'activemods'}}, 'G_' . $_;
      } else {
        push @{$atm{$mode}{'inactivemods'}}, 'G_' . $_;
      }
    }
  } # foreach ( qw/kotor1 kotor2/ ) {

  $mode = $prefmode;
  &listfill('both');

  if ($kotordir{'kotor2'} ne "") {
    if (! -d $kotordir{'kotor2'} . 'Override') {
      print("Kotor 2 Override does not exist, creating.\n");
      system("mkdir " . chr(34) . $kotordir{'kotor2'} . 'Override' . chr(34));
    } else {
      print("Kotor 2 Override exists\n");
    }
  }
  
}

sub changemode {
  &listfill('both');
}

sub writegroups {
  my $group;
  my $mod;
  my $argh;

  if ($mode eq 'kotor1') {
    $argh = 'modgroups.kmm';
  } else {
    $argh = 'modgroups2.kmm';
  }
  
  open(KMMGROUPOUT, ">", $workingdir . "\\" . $argh) or die "can't open $argh\n";

  foreach $group ( keys %{$atm{$mode}{'modgroups'}} ) {
    print(KMMGROUPOUT "<group>=" . $atm{$mode}{'modgroups'}{$group}{'name'} . "\n");
    foreach $mod ( keys %{ $atm{$mode}{'modgroups'}{$group}{'mods'} } ) {
      print(KMMGROUPOUT "<mod>=" . $atm{$mode}{'modindex'}[$mod] . "\n");
    }
    print(KMMGROUPOUT "<endgroup>". $atm{$mode}{'modgroups'}{$group}{'name'} . "\n");
  }
  close KMMGROUPOUT;
}

sub commitdata {
  my ($temp, $target);

  $target = $object{'dataentry'}->get;

  foreach (keys %{$atm{$mode}{'modgroups'}}) {
    if ($atm{$mode}{'modgroups'}{$_}{'name'} eq $target) {
      return -1;
    }
  }
  
  $temp = 0;
  while ( defined( $atm{$mode}{'modgroups'}{$temp} ) ) {
    $temp++;
  }
 
  $atm{$mode}{'modgroups'}{$temp} = {};
  $atm{$mode}{'modgroups'}{$temp}{'name'} = $target;
  $atm{$mode}{'modgroups'}{$temp}{'mods'} = {};
  foreach ( @{$atm{$mode}{'activemods'}} ) {
    $atm{$mode}{'modgroups'}{$temp}{'mods'}{$_} = $atm{$mode}{'modindex'}[$_];
    $atm{$mode}{'modslist'}{$_}{'status'} = $atm{$mode}{'modgroups'}{$temp}{'name'};
  }

  &writegroups;
  
  @{$atm{$mode}{'activemods'}} = ();
  push @{$atm{$mode}{'activemods'}}, "G_" . $temp;

  &writekmmim;
  
  $object{'dataentrywin'}->withdraw;
  &listfill('active');
}

sub creategroup {
  my $index;
	
  $index = $object{'activelist'}->info('children', '.');
  if ($index eq "") {return;}
  
  foreach (@{$atm{$mode}{'activemods'}}) {
    if (/G_/) {return;}
  }
  
  $object{'dataentry'}->delete('0', 'end');
  $object{'dataentrywin'}->deiconify;
  $object{'dataentrywin'}->raise;
  $object{'dataentrywin'}->focus;
}

sub inactiveclick {
  $object{'activelist'}->selectionClear;
}

sub activeclick {
  $object{'inactivelist'}->selectionClear;
}

sub setprefs {
  $kotordir{'kotor1'} = $object{'kotordirentry'}->get;
  $kotordir{'kotor2'} = $object{'kotor2direntry'}->get;
  $atm{'kotor1'}{'moddir'} = $object{'moddirentry'}->get;
  $atm{'kotor2'}{'moddir'} = $object{'mod2direntry'}->get;
  open(KMMCONFOUT, '>', $workingdir . "\\kmm.conf") or die "can't open kmm.conf\n";
  print(KMMCONFOUT "<kotordir>=" . $kotordir{'kotor1'} . "\n");
  print(KMMCONFOUT "<moddir>=" . $atm{'kotor1'}{'moddir'} . "\n");
  print(KMMCONFOUT "<kotor2dir>=" . $kotordir{'kotor2'} . "\n");
  print(KMMCONFOUT "<mod2dir>=" . $atm{'kotor2'}{'moddir'} . "\n");
  print(KMMCONFOUT "<mode>=" . $mode . "\n");
  close KMMCONFOUT;
  $object{'setprefs'}->withdraw;
  $object{'main'}->deiconify;
  $object{'main'}->raise;
  $object{'main'}->focus;
}

# change the destination of a mod file
sub destination {
  my ($index, $type, $target);
  my $temp;

  $object{'hiddenentry'}->delete('0', 'end');
  
  $index = $object{'filelist'}->curselection;
  if ($index ne "") {
    $target = $object{'filelist'}->get($index);
    $type = substr($target, 0, 3);
    if ($target =~ /(.*) <dest>=(.*)/) {
      $target = $1;
    }
    if ($type eq "MOD") {
      $object{'filelist'}->delete($index);
      &getfolder('hiddenentry', $kotordir{$mode});
      $temp = $object{'hiddenentry'}->get;
      print ($temp . "\n");
      if ($temp eq "\\") {
        $temp = $kotordir{$mode} . "override\\";
      }
      $temp = substr($temp, length($kotordir{$mode}));
      $object{'filelist'}->insert($index, $target . " <dest>=" . $temp);
    } 
  }
}

# toggle if a file is a mod file or a support file
sub modsuptoggle {
  my ($index, $type, $target);
  
  $index = $object{'filelist'}->curselection;
  if ($index ne "") {
    $target = $object{'filelist'}->get($index);
    $object{'filelist'}->delete($index);
    $type = substr($target, 0, 3);
    if ($type eq "MOD") {
      $target =~ /MOD: (.*) <dest>=.*/;
      $object{'filelist'}->insert($index, "SUP: " . $1);
    } else {
      $object{'filelist'}->insert($index, "MOD" . substr($target, 3) . " <dest>=override\\");
    }
  }
}

# open up notepad and display the mod info (if it has a readme file)
sub describemod {
  my ($target, $index);
	
  $index = $object{'activelist'}->selectionGet;
  if ($index ne "") {
    $target = $object{'activelist'}->info('data', $index);
    if ( $atm{$mode}{'modslist'}{$target}{'description'} ne "" ) {
      system $texteditor . " " . chr(34) . $atm{$mode}{'modslist'}{$target}{'modroot'} . $atm{$mode}{'modslist'}{$target}{'description'} . chr(34);
    }
    return;
  }
  
  $index = $object{'inactivelist'}->selectionGet;
  if ($index ne "") {
    $target = $object{'inactivelist'}->info('data', $index);
    if ( $atm{$mode}{'modslist'}{$target}{'description'} ne "" ) {
      system $texteditor . " " . chr(34) . $atm{$mode}{'modslist'}{$target}{'modroot'} . $atm{$mode}{'modslist'}{$target}{'description'} . chr(34);
    }
    return;
  }
}

# remove the mod from KMM
# if the mod is currently installed, then de-install it first
sub deletemod {
  my ($index, $target, $work, $work2);
  my @temp;

  $index = $object{'activelist'}->selectionGet;
  if ($index ne "") {
    # get the selected item
    $target = $object{'activelist'}->info('data', $index);
    # deactivate the mod
    &deactivatemod;
    # select the inactive mod
    $object{'inactivelist'}->selectionSet("." . $target);
    # now that the mod is selected the next block will delete it
  }

  $index = $object{'inactivelist'}->selectionGet;
  if ($index ne "") {
    # get the selected item and parse it
    $target = $object{'inactivelist'}->info('data', $index);
    if ($target =~ /G_(.*)/) {
      delete $atm{$mode}{'modgroups'}{$1};
      &writegroups;
    } else {
      # delete the mod from all groups
      foreach $work (keys %{$atm{$mode}{'modgroups'}}) {
        foreach ( keys %{$atm{$mode}{'modgroups'}{$work}{'mods'}} ) {
	  if ($target == $_) {
	    delete $atm{$mode}{'modgroups'}{$work}{'mods'}{$_};
	  }
        }
      }
      &writegroups;
      # delete from the 'installedmods.kmm' file
      $atm{$mode}{'modslist'}{$target} = undef;
      delete $atm{$mode}{'modslist'}{$target};
      &writekmmim;
      # remove the selected mod from the active list
    }
    foreach ( @{$atm{$mode}{'inactivemods'}} ) {
      if ($_ eq $target) {next;}
      push @temp, $_;
    }
    @{$atm{$mode}{'inactivemods'}} = @temp;
    # update the lists  
    &listfill('inactive');
    return;
  } # if ($index ne "")
  
}

# deactivate a mod and remove its files from override (and anywhere else they might be)
sub deactivatemod {
  my ($argument) = (@_);
  my ($index, $target, $modroot, $source, $dest, $file, $rensource, $lcfile, $lctarget);
  my ($Dsize, $Ssize, $Dmodified, $Smodified, @deactivatelist);
  my @temp;
  my @conflictlist;

  # check to see if there is a valid selection in the listbox
  $index = $object{'activelist'}->selectionGet;
  if ($index eq "" && $argument ne "all") {
    return;
  } elsif ($argument ne "all") {
    # get the selected item 
    $target = $object{'activelist'}->info('data', $index);
  }

  # check to see if we are deactivating a single mod or a group, or all
  if ($argument eq "all") {
    foreach $target (@{$atm{$mode}{'activemods'}}) {
      if ($target =~ /G_(.*)/) {
        push @{$atm{$mode}{'inactivemods'}}, $target;
        foreach (keys %{$atm{$mode}{'modgroups'}{$1}{'mods'}} ) {
          push @deactivatelist, $_;
          # add the selected mod to the inactive list
          push @{$atm{$mode}{'inactivemods'}}, $_;
        }
      } else {
        push @deactivatelist, $target;
        push @{$atm{$mode}{'inactivemods'}}, $target;
      }
    }
    @{$atm{$mode}{'activemods'}} = ();
  } elsif ($target =~ /G_(.*)/) { # a group
    push @{$atm{$mode}{'inactivemods'}}, $target;
    foreach (keys %{$atm{$mode}{'modgroups'}{$1}{'mods'}} ) {
      push @deactivatelist, $_;
      # add the selected mod to the inactive list
      push @{$atm{$mode}{'inactivemods'}}, $_;
      # remove the selected mod from the active list
      foreach ( @{$atm{$mode}{'activemods'}} ) {
        if ($_ eq $target) {next;}
        push @temp, $_;
      }
      @{$atm{$mode}{'activemods'}} = @temp;
      @temp = ();
    }
  } else { # a single mod
    $deactivatelist[0] = $target;
    # add the selected mod to the inactive list
    push @{$atm{$mode}{'inactivemods'}}, $target;
    # remove the selected mod from the active list
    foreach ( @{$atm{$mode}{'activemods'}} ) {
      if ($_ eq $target) {next;}
      push @temp, $_;
    }
    @{$atm{$mode}{'activemods'}} = @temp;
  }
 
  #check each file and see if it is ours
  foreach $target ( @deactivatelist ) {
    $modroot = $atm{$mode}{'modslist'}{$target}{'modroot'};
    print("Deactivating: " . $atm{$mode}{'modindex'}[$target] . "\n");
    foreach $file ( keys %{$atm{$mode}{'modslist'}{$target}{'modfiles'}} ) {
      $lcfile = lc($file);
      if (! defined( $modextensions{substr($file, length($file)-4)} ) ) {
        print("Skipping illegal file: $file\n");
        next;
      }
      $source = $modroot . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'sourcepath'} . $file;
      $dest = $kotordir{$mode} . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'dest'} . $file;
      if (-e $dest) {
        # get the info on the file
        (undef, undef, undef, undef, undef, undef, undef, $Dsize, undef, $Dmodified) = stat($dest);
        # get the info on our file
        (undef, undef, undef, undef, undef, undef, undef, $Ssize, undef, $Smodified) = stat($source);
        if (($Dsize == $Ssize) && ($Dmodified == $Smodified)) {
          #same name, same size, same modified time, must be the same file so delete it
          print("Removing $dest\n");
	  unlink $dest;
  	  # check if this file has a backup
  	  if (-e $kotordir{$mode} . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'dest'} . "_" . $file) {
            $rensource = chr(34) . $kotordir{$mode} . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'dest'} . "_" . $file . chr(34);
            print( "Restoring backed up file: $rensource\n");
            system ("ren " . $rensource . " " . $file);
 	  }
        } else {
          # don't know what we should do with this file, we are not managing it
 	  print("$dest does not belong to " . $atm{$mode}{'modindex'}[$target] . "\n");
  	  push @conflictlist, "$dest does not belong to " . $atm{$mode}{'modindex'}[$target];
	}
      } # if (-e $dest)
    } # foreach $file
    # update the mod status
    $atm{$mode}{'modslist'}{$target}{'status'} = 'inactive';
  } # foreach $target
  
  # report any file conflicts
  if ( defined($conflictlist[0]) ) {
    print("----------\n");
    foreach (@conflictlist) {
      print("$_\n");
    }
    unshift @conflictlist, "The following files were not removed:\n";
    &showerror(@conflictlist);
  }
  
  # update the 'installedmods.kmm' file
  &writekmmim;

  # update the lists  
  &listfill('both');

}

#activate a mod and copy its files to override and anywhere else they need to go
sub activatemod {
  my ($index, $target, $file, $modroot, $source, $dest, $rensource);
  my @temp;
  my @conflictlist;
  my ($Dsize, $Dmodified, $Ssize, $Smodified);
  my ($foundconflict, $lcfile, $lctarget, $groupname);
  my @activatelist;
  my @activecheck;

  # check to see if there is a valid selection in the listbox
  $index = $object{'inactivelist'}->selectionGet;
  if ($index eq "") {return;}

  # get the selected item
  $target = $object{'inactivelist'}->info('data', $index);

  # figure out if we are activating a group or a single mod
  if ($target =~ /G_(.*)/) { # a group
    $groupname = $atm{$mode}{'modgroups'}{$1}{'name'};
    foreach (keys %{$atm{$mode}{'modgroups'}{$1}{'mods'}} ) {
      push @activatelist, $_;
    }
  } else { # single mod
    $groupname = "";
    $activatelist[0] = $target;
  }

  # create list of already active mods
  @activecheck = ();
  foreach ( @{$atm{$mode}{'activemods'}} ) {
    if (/G_(.*)/) { # a group
      foreach (keys %{$atm{$mode}{'modgroups'}{$1}{'mods'}} ) {
        push @activecheck, $_;
      }
    } else {
      push @activecheck, $_;
    }
  }

  $foundconflict = 0;
  # check to see if the mod(s) we are activating are already active
  foreach $dest ( @activatelist ) {
    foreach ( @activecheck ) {
      if ($dest == $_) {
        $foundconflict = 1;
	push @conflictlist, $atm{$mode}{'modindex'}[$dest] . " is already active";
      }
    }
  }
  
  foreach $target ( @activatelist ) {
    $modroot = $atm{$mode}{'modslist'}{$target}{'modroot'};
    # check for file conflicts with activated mods and unmanaged files
    foreach $file ( keys %{$atm{$mode}{'modslist'}{$target}{'modfiles'}} ) {
      $lcfile = lc($file);
      # check against the other active mods
      foreach ( @activecheck ) {
        if ($_ eq $target) {next;} # don't match against the mod we are activating
        if ( defined( $atm{$mode}{'modslist'}{lc($_)}{'modfiles'}{$lcfile} ) ) {
	  push @conflictlist, "$file conflicts with file in " . $atm{$mode}{'modindex'}[$_];
  	  $foundconflict = 1;
        }
      }
      # check for unmanaged files that may conflict
      $source = $modroot . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'sourcepath'} . $file;
      $dest = $kotordir{$mode} . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'dest'} . $file;
      if (-e $dest && $foundconflict == 0) {
        # get the info on the conflicting file
        (undef, undef, undef, undef, undef, undef, undef, $Dsize, undef, $Dmodified) = stat($dest);
        # get the info on our file
        (undef, undef, undef, undef, undef, undef, undef, $Ssize, undef, $Smodified) = stat($source);
        if (($Dsize == $Ssize) && ($Dmodified == $Smodified)) {
          #same name, same size, same modified time, must be the same file
  	  # delete it so we can put it back in the next loop, silly huh?
          print("Removing $dest\n");
  	  unlink $dest;
        } else {
          # don't know what we should do with this file, we are not managing it
  	  push @conflictlist, "$dest already exists.";
        }
      }
      # $foundconflict = 0;
    } # foreach $file
  } # foreach $target
  if ( defined($conflictlist[0]) ) {
    print("----------\n");
    foreach (@conflictlist) {
      print("$_\n");
    }
    &showerror(@conflictlist);
    return;
  }

  $target = $object{'inactivelist'}->info('data', $index);
  # we are good to go, add the group to the active list
  push @{$atm{$mode}{'activemods'}}, $target;
  # remove the group from the inactive list
  @temp = ();
  foreach ( @{$atm{$mode}{'inactivemods'}} ) {
    if ($_ eq $target) {next;}
    push @temp, $_;
  }
  @{$atm{$mode}{'inactivemods'}} = @temp;
  
  # now add each mod to the active list
  foreach $target ( @activatelist ) {
    $modroot = $atm{$mode}{'modslist'}{$target}{'modroot'};
    #copy the files to their locations
    foreach $file ( keys %{$atm{$mode}{'modslist'}{$target}{'modfiles'}} ) {
      $lcfile = lc($file);
      print("----------------------\n");
      if (! defined( $modextensions{ lc(substr($file, length($file)-4)) } ) ) {
        print("Skipping illegal file: $file\n");
        next;
      }
      $source = chr(34) . $modroot . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'sourcepath'} . $file . chr(34);
      $dest = chr(34) . $kotordir{$mode} . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'dest'} . chr(34);
      $dest = substr($dest, 0, -1) . chr(34); # hack for win98 users.  Yes win98 users!
      if (-e $kotordir{$mode} . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'dest'} . $file) {
        $rensource = chr(34) . $kotordir{$mode} . $atm{$mode}{'modslist'}{$target}{'modfiles'}{$lcfile}{'dest'} . $file . chr(34);
        print( $rensource . " exists, making backup\n");
        system ("ren " . $rensource . " _" . $file);
        print( "copy " . $source . " " . $dest . "\n");
        system ("copy " . $source . " " . $dest);
      } else {
        print( "copy " . $source . " " . $dest . "\n");
        system ("copy " . $source . " " . $dest);
      }
    }
    # add the selected mod to the active list
    # push @{$atm{$mode}{'activemods'}}, $target;
    # remove the selected mod from the inactive list
    @temp = ();
    foreach ( @{$atm{$mode}{'inactivemods'}} ) {
      if ($_ eq $target) {next;}
      push @temp, $_;
    }
    @{$atm{$mode}{'inactivemods'}} = @temp;
  
    # update the 'installedmods.kmm' file
    if ($groupname ne "") {
      $atm{$mode}{'modslist'}{$target}{'status'} = $groupname;
    } else {
      $atm{$mode}{'modslist'}{$target}{'status'} = 'active';
    }
  } # foreach $target
  &writekmmim;
  
  # update the lists  
  &listfill('both');
}

# write out a kmm file for the mod
sub writekmm {
  my ($type, $filename, $filepath, $ext, $size, $modified, $dest);
  my ($temp, $index);
  my $modname = $object{'modnameentry'}->get;
  my $modroot = $object{'modpathentry'}->get;

  if ( defined($atm{$mode}{'modlookup'}{$modname}) ) {
    $index = $atm{$mode}{'modlookup'}{$modname};
  } else {
    $index = 0;
    while (defined( $atm{$mode}{'modslist'}{$index} ) ) {
      $index++;
    }
    $atm{$mode}{'modindex'}[$index] = $modname;
    $atm{$mode}{'modlookup'}{$modname} = $index;
  }

  $atm{$mode}{'modslist'}{$index} = {};
  $atm{$mode}{'modslist'}{$index}{'modroot'} = $modroot;
  # create the kmm file
  open(KMMOUT, ">", $modroot . "$modname.kmm") or die "can't open out file\n";
  print("writing: " . $modroot . "$modname.kmm\n");
  # the name of the mod
  print(KMMOUT "<mod>=" . $modname . "\n");
  # what version of kotor the mod is for
  print(KMMOUT "<kotor>=" . $mode . "\n");
  # the author of the mod
  $atm{$mode}{'modslist'}{$index}{'author'} = $object{'modauthorentry'}->get;
  print(KMMOUT "<author>=" . $object{'modauthorentry'}->get . "\n");
  # the version number of the mod
  $atm{$mode}{'modslist'}{$index}{'version'} = $object{'modverentry'}->get;
  print(KMMOUT "<version>=" . $object{'modverentry'}->get . "\n");
  # mod description file, usually: "d:\moddir\readme.txt"
  $atm{$mode}{'modslist'}{$index}{'description'} = $object{'moddescfileentry'}->get;
  print(KMMOUT "<readme>=" . $object{'moddescfileentry'}->get . "\n");

  foreach (sort $object{'filelist'}->get('0', 'end')) {
    if (/\.kmm/) {next;} # skip kmm files
    # check if the file has a non-override destination
    # if it does chop it off and store it
    if (/(.*) <dest>=(.*)/) {
      $temp = $1;
      $dest = $2;
    } else {
      $dest = 'override\\';
      $temp = $_;
    }
      
    # check if this file is in a subdirectory
    if ( index($temp, '\\') > -1) {
      # the file is in a subdirectory
      $temp =~ /(SUP:|MOD:) (.*\\)(.*)(....)/;
      $type = $1;
      $filepath = $2;
      $filename = $3;
      $ext = $4;
    } else {
      # the file is not in a subdirectory
      $temp =~ /(SUP:|MOD:) (.*)(....)/;
      $type = $1;
      $filepath = "";
      $filename = $2;
      $ext = $3;
    }

    # if this is a mod file add it to the modfiles list
    if ($type eq "MOD:") {
      $atm{$mode}{'modslist'}{$index}{'modfiles'}{lc($filename . $ext)}{'dest'} = $dest;
      $atm{$mode}{'modslist'}{$index}{'modfiles'}{lc($filename . $ext)}{'sourcepath'} = $filepath;
      print(KMMOUT "<modfile>=$filepath,$filename$ext,$dest\n");     
    } else {
      # the file is not a mod file, so it must be a support file
      $atm{$mode}{'modslist'}{$index}{'supportfiles'}{$filename . $ext} = $modroot . $filepath;	
      print(KMMOUT "<supportfile>=$filepath$filename$ext\n");     
    }
  }
  # add an entry for this kmm file we are writing
  print(KMMOUT "<supportfile>=$modname.kmm\n");     

  close KMMOUT;
}

# clear out the add mod window
sub cleanoutaddmod {
  # clear out the entry boxes
  $object{'modpathentry'}->delete('0', 'end');
  $object{'modverentry'}->delete('0', 'end');
  $object{'modauthorentry'}->delete('0', 'end');
  $object{'moddescfileentry'}->delete('0', 'end');
  # clear out the listbox
  $object{'filelist'}->delete('0', 'end');
}

# actually add the mod to the KMM mod list
sub commitaddmod {
  my ($arg) = (@_);
  my ($temp, $index);
  my $modname = $object{'modnameentry'}->get;
  my $root = $object{'modpathentry'}->get;

  # check if there is a kmm file.  If there isn't one create one
  print("looking for " . $root . $modname . ".kmm\n");
  if(! -e $root . $modname . ".kmm") { # no kmm file, so create it
    print("no kmm file, creating one\n");
    &writekmm;
  } elsif ((-e $root . $modname . ".kmm") && ($arg eq 'update')) { #kmm file exists, but an update was requested
    print("kmm file exists, force update requested\n");
    &writekmm;
  } else { # kmm file found, no update requested
    print($root . $modname . ".kmm exists, leaving it alone\n");
  }

  $index = $atm{$mode}{'modlookup'}{$modname};
  
  # add the mod to the 'installedmods.kmm' file
  $atm{$mode}{'modslist'}{$index}{'kmmfile'} = $root . $modname . ".kmm";
  $atm{$mode}{'modslist'}{$index}{'status'} = 'inactive';
  &writekmmim;

  #add the new mod to the inactive list
  push @{$atm{$mode}{'inactivemods'}}, $index;
  
  # update the inactive list
  &listfill('inactive');
  
  # clear out the add mod window
  &cleanoutaddmod;
  # hide the add mod window
  $object{'addmodwin'}->withdraw;
}

# write the KMM installed mods file
sub writekmmim {
  my $argh;

  if ($mode eq 'kotor1') {
    $argh = 'installedmods.kmm';
  } else {
    $argh = 'installedmods2.kmm';
  }
	
  open(KMMIMOUT, ">", $workingdir . "\\" . $argh) or die "can't open $argh\n";
  foreach ( keys %{$atm{$mode}{'modslist'}} ) {
    print(KMMIMOUT $atm{$mode}{'modslist'}{$_}{'kmmfile'} . "," . $atm{$mode}{'modslist'}{$_}{'status'} . "\n");
  }
  close KMMIMOUT;
}

# the sort sub
sub kmmsort {
  if ($a =~ /G_/ && $b =~ /G_/) {
    return $atm{$mode}{'modgroups'}{substr($a,2)}{'name'} cmp $atm{$mode}{'modgroups'}{substr($b,2)}{'name'};
  } elsif ($a =~ /G_/) {
    return -1;
  } elsif ($b =~ /G_/) {
    return 1;
  } else {
    return lc($atm{$mode}{'modindex'}[$a]) cmp lc($atm{$mode}{'modindex'}[$b]);
  }
}

# fill and sort the active and inactive listboxes
sub listfill {
  my ($tofill) = (@_);
  my ($curmod, $count, $groupnum);

  if ($tofill eq 'both' or $tofill eq 'active') {
    print($mode . "\n");
    @{$atm{$mode}{'activemods'}} = sort kmmsort @{$atm{$mode}{'activemods'}};
    $object{'activelist'}->delete('all');
    $object{'activelist'}->add('.', 
	                         -text => "active mods",
                                 -data => "active");
    # first we will look at the active mods list
    foreach $curmod ( @{$atm{$mode}{'activemods'}} ) {
      if ($curmod =~ /G_(.*)/) { # this is an active group
	$groupnum = $1;
	# add the group root
        $object{'activelist'}->add(".G_$curmod", 
                                   -text => "G_" . $atm{$mode}{'modgroups'}{$groupnum}{'name'},
                                   -data => $curmod);
	# add the mods to the group
        foreach (sort {lc($atm{$mode}{'modindex'}[$a]) cmp lc($atm{$mode}{'modindex'}[$b])} keys %{$atm{$mode}{'modgroups'}{$groupnum}{'mods'}} ) {
          $object{'activelist'}->add(".G_$curmod.$_", 
	                           -text => $atm{$mode}{'modindex'}[$_],
			           -data => $curmod);
        }
        $object{'activelist'}->setmode(".G_$curmod", "close");
        $object{'activelist'}->close(".G_$curmod");
      } else { # this is a mod not in a group
	# add the mod root, and base info
        $object{'activelist'}->add(".$curmod", 
                                   -text => $atm{$mode}{'modindex'}[$curmod] . " (ver: " . $atm{$mode}{'modslist'}{$curmod}{'version'} . ")",
                                   -data => $curmod);
        $object{'activelist'}->add(".$curmod.author", 
  	                         -text => "author: " . $atm{$mode}{'modslist'}{$curmod}{'author'},
  			         -data => $curmod);
        $object{'activelist'}->add(".$curmod.files", 
  	                         -text => "files",
  			         -data => $curmod);
        $count = 0;
	# add the mod files list
        foreach (sort keys %{$atm{$mode}{'modslist'}{$curmod}{'modfiles'}} ) {
          $object{'activelist'}->add(".$curmod.files.$count", 
	                           -text => "MOD: " . $_,
			           -data => $curmod);
          $count++;
        }
	# add the support files list
        foreach (sort keys %{$atm{$mode}{'modslist'}{$curmod}{'supportfiles'}} ) {
          $object{'activelist'}->add(".$curmod.files.$count", 
   	                           -text => "SUP: " . $_,
  			           -data => $curmod);
  	  $count++;
        }
        $object{'activelist'}->setmode(".$curmod.files", "close");
        $object{'activelist'}->close(".$curmod.files");
        $object{'activelist'}->setmode(".$curmod", "close");
        $object{'activelist'}->close(".$curmod");
      } # if $curmod =~ /g(.*)/
    } # foreach $curmod ( @{$atm{$mode}{'activemods'}} )
  }
  
  # now we look at the inactive list
  if ($tofill eq 'both' or $tofill eq 'inactive') {
    @{$atm{$mode}{'inactivemods'}} = sort kmmsort @{$atm{$mode}{'inactivemods'}};
    $object{'inactivelist'}->delete('all');
    $object{'inactivelist'}->add('.', 
	                         -text => "Inactive mods",
                                 -data => "inactive");
    foreach $curmod ( @{$atm{$mode}{'inactivemods'}} ) {
      if ($curmod =~ /G_(.*)/) { # this is a group
	$groupnum = $1;
	# create the group root
        $object{'inactivelist'}->add(".G_$curmod", 
                                   -text => "G_" . $atm{$mode}{'modgroups'}{$groupnum}{'name'},
                                   -data => $curmod);
	# add the mods list to the group
        foreach (sort {lc($atm{$mode}{'modindex'}[$a]) cmp lc($atm{$mode}{'modindex'}[$b])} keys %{$atm{$mode}{'modgroups'}{$groupnum}{'mods'}} ) {
          $object{'inactivelist'}->add(".G_$curmod.$_", 
	                           -text => $atm{$mode}{'modindex'}[$_],
			           -data => $curmod);
        }
        $object{'inactivelist'}->setmode(".G_$curmod", "close");
        $object{'inactivelist'}->close(".G_$curmod");
      } else { # this is a mod not in a group
	# add the mod root and base info
        $object{'inactivelist'}->add(".$curmod", 
                                   -text => $atm{$mode}{'modindex'}[$curmod] . " (ver: " . $atm{$mode}{'modslist'}{$curmod}{'version'} . ")",
                                   -data => $curmod);
        $object{'inactivelist'}->add(".$curmod.author", 
	                           -text => "author: " . $atm{$mode}{'modslist'}{$curmod}{'author'},
			           -data => $curmod);
        $object{'inactivelist'}->add(".$curmod.files", 
	                           -text => "files",
			           -data => $curmod);
        $count = 0;
	# add the mod files to the mod
        foreach (sort keys %{$atm{$mode}{'modslist'}{$curmod}{'modfiles'}} ) {
          $object{'inactivelist'}->add(".$curmod.files.$count", 
	                           -text => "MOD: " . $_,
			           -data => $curmod);
	  $count++;
        }
	# add the support files to the mod
        foreach (sort keys %{$atm{$mode}{'modslist'}{$curmod}{'supportfiles'}} ) {
          $object{'inactivelist'}->add(".$curmod.files.$count", 
	                           -text => "SUP: " . $_,
	  		           -data => $curmod);
	  $count++;
        }
        $object{'inactivelist'}->setmode(".$curmod.files", "close");
        $object{'inactivelist'}->close(".$curmod.files");
        $object{'inactivelist'}->setmode(".$curmod", "close");
        $object{'inactivelist'}->close(".$curmod");
      }
    }
  }
}

# used to find the files when adding a new mod without a KMM file
sub wanted {
  my $root = $object{'modpathentry'}->get;
  my $temp = $File::Find::name;
  my $ext = substr($temp, length($temp)-4);

  if (-f $temp) {
    $temp =~ s#\\/#\\#g;
    $temp =~ s#/#\\#g;
    $temp = substr($temp, length($root));
    if ( defined( $modextensions{lc($ext)} ) ) {
      $object{'filelist'}->insert('end', "MOD: " . $temp . " <dest>=override\\");
    } else {
      $object{'filelist'}->insert('end', "SUP: " . $temp);
    }
  }
}

# start the add mod process
sub addmod {
  my ($temp, $loc, $kmmfile, $readme);
  my $goodmod = 1;
	
  # clear out the add mod window
  &cleanoutaddmod;
  
  #open the file dialog and let the user look for a mod dir
  if (&getfilepath('modpathentry', 'dironly', $atm{$mode}{'moddir'}) != 1) {return -1;}
    
  # open the selected dir and look for a .kmm file
  my $modroot = $object{'modpathentry'}->get;
  opendir MODDIR, $modroot;
  foreach (readdir MODDIR) {
    if ( /\.kmm/ ) {
      $kmmfile = $modroot . $_;
      last;
    }
  }
  closedir MODDIR;

  # if we found a .kmm file check it out, else collect the mod info the hard way
  if ($kmmfile ne "") {
    print("kmm file '$kmmfile' found\n");
    foreach ( keys %{$atm{$mode}{'modslist'}} ) {
      if( $atm{$mode}{'modslist'}{$_}{'kmmfile'} eq $kmmfile ) {
        print("That mod is already installed!\n");
	&showerror("That mod is already installed!");
	return;
      }
    }
    $goodmod = &readkmm($kmmfile, $modroot, 'addmod');
  } else {
    print("no kmm file found\n");
    $temp = $object{'modpathentry'}->get;
    $loc = rindex($temp, "\\", length($temp)-2)+1;
    $temp = substr($temp, $loc, (length($temp)-$loc)-1);
    $object{'modnameentry'}->delete('0', 'end');
    $object{'modnameentry'}->insert('end', $temp);

    $object{'filelist'}->delete('0', 'end');

    # look for a file with 'readme' in the name
    opendir MODDIR, $modroot;
    foreach (readdir MODDIR) {
      if ( /readme/i ) {
        $object{'moddescfileentry'}->delete('0', 'end');
        $object{'moddescfileentry'}->insert('end', $_);
        system "start notepad $modroot$_";
        last;
      }
    }
    closedir MODDIR;
  
    find(\&wanted, $object{'modpathentry'}->get);
  }
  
  if ($goodmod == 1) {
    $object{'addmodwin'}->deiconify;
    $object{'addmodwin'}->raise;
    $object{'addmodwin'}->focus;
  }
}

# read a KMM file
sub readkmm {
  my ($kmmfile, $modroot, $caller) = (@_);
  my ($modname, $index, $thismode);

  open(KMMIN, $kmmfile) or die "can't open kmm file\n";
  print("Finding version of $kmmfile\n");
  while (<KMMIN>) {
    if (/<kotor>=(.*)/) {
      $thismode = $1;
      last;
    }
  }
	  
  if ($thismode eq "") {
    $thismode = 'kotor1';
  }

  print("Mod is for $thismode\n");
  
  if ($thismode eq 'kotor1' && $mode eq 'kotor2') {
    print("This mod is for Kotor 1!\n");
    return -1;
  } elsif ($thismode eq 'kotor2' && $mode eq 'kotor1') {
    print("This mod is for Kotor 2!\n");
    return -1;
  }
  close KMMIN;
  
  open(KMMIN, $kmmfile) or die "can't open kmm file\n";
  print("reading $kmmfile\n");
  
  while (<KMMIN>) {
    if (/<mod>=(.*)/) {
      push @{$atm{$mode}{'modindex'}}, $1;
      #$index = $#modindex;
      $index = $#{$atm{$mode}{'modindex'}};
      $atm{$mode}{'modlookup'}{$1} = $index;
      $modname = $1;
    } elsif (/<author>=(.*)/) {
      $atm{$mode}{'modslist'}{$index}{'author'} = $1;
    } elsif (/<version>=(.*)/) {
      $atm{$mode}{'modslist'}{$index}{'version'} = $1;
    } elsif (/<readme>=(.*)/) {
      $atm{$mode}{'modslist'}{$index}{'description'} = $1;
    } elsif (/<modfile>=(.*),(.*),(.*)/) {
      $atm{$mode}{'modslist'}{$index}{'modfiles'}{lc($2)}{'sourcepath'} = $1;	
      $atm{$mode}{'modslist'}{$index}{'modfiles'}{lc($2)}{'dest'} = $3;	
    } elsif (/<supportfile>=(.*)/) {
      $atm{$mode}{'modslist'}{$index}{'supportfiles'}{$1} = $modroot . $1;	
    }
  }
  close KMMIN;

  
  $atm{$mode}{'modslist'}{$index}{'kmmfile'} = $kmmfile;
  #$atm{$mode}{'modslist'}{$index}{'status'} = ;
  
  $atm{$mode}{'modslist'}{$index}{'modroot'} = $modroot;
  if ($caller eq "addmod") {
    $object{'modnameentry'}->delete('0', 'end');
    $object{'modnameentry'}->insert('end', $modname);

    $object{'modauthorentry'}->delete('0', 'end');
    $object{'modauthorentry'}->insert('0', $atm{$mode}{'modslist'}{$index}{'author'});

    $object{'modverentry'}->delete('0', 'end');
    $object{'modverentry'}->insert('0', $atm{$mode}{'modslist'}{$index}{'version'});

    $object{'moddescfileentry'}->delete('0', 'end');
    $object{'moddescfileentry'}->insert('0', $atm{$mode}{'modslist'}{$index}{'description'});

    foreach ( sort keys %{$atm{$mode}{'modslist'}{$index}{'modfiles'}} ) {
      $object{'filelist'}->insert('end', "MOD: " . $atm{$mode}{'modslist'}{$index}{'modfiles'}{$_}{'sourcepath'} . $_ . " <dest>=" . $atm{$mode}{'modslist'}{$index}{'modfiles'}{$_}{'dest'});
    }

    foreach ( sort keys %{$atm{$mode}{'modslist'}{$index}{'supportfiles'}} ) {
      $object{'filelist'}->insert('end', "SUP: " . $_);
    }
  } elsif ($caller eq 'init') {
  }
  return 1;
}

# open a window to select a folder
sub getfolder {
  my ($target, $start) = (@_);

  $object{$target}->delete('0', 'end');
  $object{$target}->insert('end', Win32::GUI::BrowseForFolder(-root => $start) . "\\");
  
}

# open a window to select a file
sub getfilepath {
  my ($entry, $type, $startdir) = (@_);
  my $temp;

  print("startdir: " . $startdir . "\n");
  
  my %parms=(title => "Select mod directory", 
	     handle=> 0,
             dir => $startdir,
             filters => { 'All Files' => '*.*'},
             options => OFN_PATHMUSTEXIST);
      
  $temp = OpenDialog \%parms;
  if ($type eq "dironly") {
    $temp =~ /(.*\\)(.*\..*)/;
    $temp = $1;
  }

  if (-e $temp) {
    $object{$entry}->delete('0','end');
    $object{$entry}->insert('end',$temp);
    return 1;
  } else {
    print ("Path/file does not exist!\n");
    return -1;
  }
}

# routine to hide a window instead of destroying it
sub withdrawwindow {
  my ($target) = (@_);
  
  # hide the window
  $object{$target}->withdraw;

}

sub showerror {
  my (@messagelist) = (@_);
  my $message;

  if ( defined($messagelist[1]) ) {
    $message = "The following errors occured:\n\n"; 
  } else {
    $message = "The following error occured:\n\n";
  }

  foreach (@messagelist) {
    $message .= $_ . "\n";
  }
  
  $object{'main'}->messageBox(-message => $message, 
	                      -title => "KMM status", 
			      -type => 'OK');

}

