
#findrefs.pl -- finds references to scripts within various GFF files
#v0.1 - Oct 4, 2004 - NEW!
#v0.2 - Oct 4, 2004 - Added previewing (file slurps) to increase speed
#v0.3 - Oct 7, 2004 - Added .uti,case sensitivity,partial matching options(-ICP), Tag field searches
#v0.4 - Oct 7, 2004 - Friendlier output.  -q option added

use Getopt::Long qw(:config no_ignore_case bundling);
use Win32::TieRegistry;
use Bioware::GFF;
use strict;
my $version ="0.4";

sub search_rim;
sub search_ifo;
sub search_are;
sub search_utt;
sub search_utp;
sub search_ncs;
sub search_utm;
sub search_utc;
sub search_dlg;
sub search_utd;
sub search_bif23;
sub search_bif21;

my %resource_types =(
0x0000 => 'res', 	#Misc. GFF resources
0x0001 => 'bmp', 	#Microsoft Windows Bitmap
0x0002 => 'mve',
0x0003 => 'tga', 	#Targa Graphics Format
0x0004 => 'wav', 	#Wave
0x0006 => 'plt', 	#Bioware Packed Layer Texture
0x0007 => 'ini', 	#Windows INI
0x0008 => 'mp3', 	#MP3
0x0009 => 'mpg', 	#MPEG
0x000A => 'txt', 	#Text file
0x000B => 'wma', 	#Windows Media audio?
0x000C => 'wmv', 	#Windows Media video?
0x000D => 'xmv',
0x07D0 => 'plh',
0x07D1 => 'tex',
0x07D2 => 'mdl', 	#Model
0x07D3 => 'thg',
0x07D5 => 'fnt', 	#Font
0x07D7 => 'lua',
0x07D8 => 'slt',
0x07D9 => 'nss', 	#NWScript source code
0x07DA => 'ncs', 	#NWScript bytecode
0x07DB => 'mod', 	#Module
0x07DC => 'are', 	#Area (GFF)
0x07DD => 'set', 	#Tileset (unused in KOTOR?)
0x07DE => 'ifo', 	#Module information
0x07DF => 'bic', 	#Character sheet (unused)
0x07E0 => 'wok', 	# walk-mesh
0x07E1 => '2da', 	#2-dimensional array
0x07E2 => 'tlk', 	#conversation file
0x07E6 => 'txi', 	#Texture information
0x07E7 => 'git', 	#Dynamic area information, game instance file, all area and objects that are scriptable
0x07E8 => 'bti',
0x07E9 => 'uti', 	#item blueprint
0x07EA => 'btc',
0x07EB => 'utc', 	#Creature blueprint
0x07ED => 'dlg', 	#Dialogue
0x07EE => 'itp', 	#tile blueprint pallet file
0x07EF => 'btt',
0x07F0 => 'utt', 	#trigger blueprint
0x07F1 => 'dds', 	#compressed texture file
0x07F2 => 'bts',
0x07F3 => 'uts', 	#sound blueprint
0x07F4 => 'ltr', 	#letter combo probability info
0x07F5 => 'gff', 	#Generic File Format
0x07F6 => 'fac', 	#faction file
0x07F7 => 'bte',
0x07F8 => 'ute', 	#encounter blueprint
0x07F9 => 'btd',
0x07FA => 'utd', 	#door blueprint
0x07FB => 'btp',
0x07FC => 'utp', 	#placeable object blueprint
0x07FD => 'dft', 	#default values file (text-ini)
0x07FE => 'gic', 	#game instance comments
0x07FF => 'gui', 	#GUI definition (GFF)
0x0800 => 'css',
0x0801 => 'ccs',
0x0802 => 'btm',
0x0803 => 'utm', 	#store merchant blueprint
0x0804 => 'dwk', 	#door walkmesh
0x0805 => 'pwk', 	#placeable object walkmesh
0x0806 => 'btg',
0x0807 => 'utg',
0x0808 => 'jrl', 	#Journal
0x0809 => 'sav', 	#Saved game (ERF)
0x080A => 'utw', 	#waypoint blueprint
0x080B => '4pc',
0x080C => 'ssf', 	#sound set file
0x080D => 'hak', 	#Hak pak (unused)
0x080E => 'nwm',
0x080F => 'bik', 	#movie file (bik format)
0x0810 => 'ndb',        #script debugger file
0x0811 => 'ptm',        #plot manager/plot instance
0x0812 => 'ptt',        #plot wizard blueprint
0x0BB8 => 'lyt',
0x0BB9 => 'vis',
0x0BBA => 'rim', 	#See RIM File Format
0x0BBB => 'pth', 	#Path information? (GFF)
0x0BBC => 'lip',
0x0BBD => 'bwm',
0x0BBE => 'txb',
0x0BBF => 'tpc', 	#Texture
0x0BC0 => 'mdx',
0x0BC1 => 'rsv',
0x0BC2 => 'sig',
0x0BC3 => 'xbx',
0x270D => 'erf', 	#Encapsulated Resource Format
0x270E => 'bif',
0x270F => 'key'
                 );
my %resource_numbers=reverse %resource_types;
my %inf = (
    'ifo'=>{'sought'=>0,'code'=>\&search_ifo},
    'are'=>{'sought'=>0,'code'=>\&search_are},
    'utc'=>{'sought'=>0,'code'=>\&search_utc},
    'uti'=>{'sought'=>0,'code'=>\&search_uti},
    'utd'=>{'sought'=>0,'code'=>\&search_utd},
    'utm'=>{'sought'=>0,'code'=>\&search_utm},
    'utp'=>{'sought'=>0,'code'=>\&search_utp},
    'utt'=>{'sought'=>0,'code'=>\&search_utt},
    'ncs'=>{'sought'=>0,'code'=>\&search_ncs},
    'dlg'=>{'sought'=>0,'code'=>\&search_dlg},
    'uti'=>{'sought'=>0,'code'=>\&search_uti});
#-----------------------------------------------------------------------
#  Main
#-----------------------------------------------------------------------

my ($opt_all, $opt_help, $opt_case_sensitive, $opt_allow_partial_matches, $opt_quiet);
# decide what user wants to do
my $result = GetOptions ("i"=>\$inf{'ifo'}{'sought'},
                         "a"=>\$inf{'are'}{'sought'},
                         "c"=>\$inf{'utc'}{'sought'},
                         "d"=>\$inf{'utd'}{'sought'},
                         "m"=>\$inf{'utm'}{'sought'},
                         "D"=>\$inf{'dlg'}{'sought'},
                         "A"=>\$opt_all,
                         "p"=>\$inf{'utp'}{'sought'},
                         "t"=>\$inf{'utt'}{'sought'},
                         "n"=>\$inf{'ncs'}{'sought'},
                         "I"=>\$inf{'uti'}{'sought'},
                         "P"=>\$opt_allow_partial_matches,
                         "C"=>\$opt_case_sensitive,
                         "q"=>\$opt_quiet,
                         "help|?"=>\$opt_help
                        );
if (($opt_help) or ($#ARGV==-1)){
    print "  __ _         _          __\n";
    print " / _(_)_ _  __| |_ _ ___ / _|___\n";
    print "|  _| | ' \\/ _` | '_/ -_)  _(_-< v$version\n";
    print "|_| |_|_||_\\__,_|_| \\___|_| /__/ by tk102\n";
    print "\n";
    print "  Purpose: Locates references to scripts/globals within .gff/.ncs file types\n";
    print "  Usage:\n";
    print "    findrefs <options> <searchstring>\n";
    print "  where <searchstring> is a script reference or a global variable\n";
    print "  Valid options (can be combined):\n";
    printf ("    %-35s%s","-a: search .are files (areas)","-A: search ALL file types\n");
    printf ("    %-35s%s","-c: search .utc files (creatures)","-C: search is CASE sensitive\n");
    printf ("    %-35s%s","-D: search .dlg files (dialogs)","-P: search allows PARTIAL matches\n");
    printf ("    %-35s%s","-d: search .utd files (doors)","-q: quiet mode\n",);
    printf ("    %-35s%s","-I: search .uti files (items)","-?: shows usage\n");
    printf ("    %-35s%s","-i: search .ifo files (modules)","\n");
    printf ("    %-35s%s","-m: search .utm files (merchants)","\n");
    printf ("    %-35s%s","-n: search .ncs files (scripts)","\n");
    printf ("    %-35s%s","-p: search .utp files (placeables)","\n");
    printf ("    %-35s%s","-t: search .utt files (triggers)","\n");
    print "\n";
    print " Example:\n";
    print "    findrefs -Dni k_pdan_bast12\n";
    exit;
}

if ($opt_all) {
    for my $key (keys %inf) {
        $inf{$key}{'sought'}=1;
    }
}


my $scriptref=$ARGV[0];
my $searchterm=$scriptref;
unless ($opt_allow_partial_matches) {$searchterm='\b'.$searchterm.'\b'}
unless ($opt_case_sensitive) {$searchterm='(?i)'.$searchterm}

#find users directories

my $registered_path;
eval { my  $kotor_key= new Win32::TieRegistry "LMachine/Software/Bioware/SW/Kotor",             #read registry
          {Access=>Win32::TieRegistry::KEY_READ, Delimiter=>"/"};
        $registered_path= $kotor_key->GetValue("Path") };
if ($@) { die "Could not locate SW KOTOR installation directory" }
unless (opendir SAVDIR, $registered_path."/modules") {
    die "Could not locate 'modules' directory for SW KOTOR"
}
close SAVDIR;
unless (opendir SAVDIR, $registered_path."/data") {
    die "Could not locate 'data' directory for SW KOTOR"
}
close SAVDIR;
chdir $registered_path;
unless (-e "chitin.key") {
    die "Could not locate chitin.key"
}

# get list of rim files
chdir "$registered_path/modules";
my @all_rims=glob("*.rim");
my @s_rims;
my @rims;
for my $rim (@all_rims) {
    if ($rim =~/_s\.rim/) {
        push @s_rims,$rim;
    } else {
        push @rims,$rim;
    }
}
my $oldrim_len=0;
if ( $inf{'utc'}{'sought'} || $inf{'utd'}{'sought'} || $inf{'utm'}{'sought'} || $inf{'dlg'}{'sought'} ||  $inf{'ncs'}{'sought'} || $inf{'utt'}{'sought'} || $inf{'uti'}{'sought'}) {
    for my $rim (@s_rims) {
        unless ($opt_quiet) {
            my $backspace=v8 x $oldrim_len;
            my $rimtxt="\[$rim\]      ";
            $oldrim_len=length $rimtxt;
            syswrite STDOUT,$backspace.$rimtxt;
        }
        search_rim($rim);
    }
}
if ($inf{'ifo'}{'sought'} || $inf{'are'}{'sought'}) {
    for my $rim (@rims) {
        unless ($opt_quiet) {
            my $backspace=v8 x $oldrim_len;
            my $rimtxt="\[$rim\]      ";
            $oldrim_len=length $rimtxt;
            syswrite STDOUT,$backspace.$rimtxt;
        }
        search_rim($rim);
    }
}


if ( $inf{'utc'}{'sought'} || $inf{'utd'}{'sought'} || $inf{'utm'}{'sought'} || $inf{'dlg'}{'sought'} ||  $inf{'utt'}{'sought'} || $inf{'uti'}{'sought'}) {
    chdir $registered_path;
    unless ($opt_quiet) {
        my $backspace=v8 x $oldrim_len;
        my $rimtxt="[templates.bif]      ";
        $oldrim_len=length $rimtxt;
        syswrite STDOUT,$backspace.$rimtxt;
    }
    search_bif23();
}
if ( $inf{'ncs'}{'sought'}) {
    chdir $registered_path;
    unless ($opt_quiet) {
        my $backspace=v8 x $oldrim_len;
        my $rimtxt="[scripts.bif]      ";
        $oldrim_len=length $rimtxt;
        syswrite STDOUT,$backspace.$rimtxt;
    }
    search_bif21();
}
unless ($opt_quiet) {
my $backspace=v8 x $oldrim_len;
syswrite STDOUT,$backspace."...Finished      ";
}






sub search_rim {
    my $rim=shift;
    my $rimfh;
    unless (open $rimfh,"<",$rim) {
        print "\nCould not open $rim\n";
        $oldrim_len=0;
        return;
    }
    sysread $rimfh, my ($sig), 8;
    unless ($sig eq "RIM V1.0") {
        close $rimfh;
        print "\n$rim not valid.\n";
        $oldrim_len=0;
        return;
    }
    {  #this bit adds speed... it previews the rim and skips it if the string isn't found
     local $/=\32768;                             # go into slurp mode (32768 byte chunks)
     my $foundit=0;
     my $rim_chunk;
     my $last_chunk;
     while (my $this_chunk=<$rimfh>) {             # slurp chunks
        $rim_chunk=$this_chunk.$last_chunk;        # just in case string is split by chunk boundary
        if ($rim_chunk=~/$searchterm/) {
            $foundit=1;
            last;
        }
        $last_chunk=$this_chunk;
     }
     return unless ($foundit);     # take a peek
    }
    sysseek $rimfh, 12, 0;
    sysread $rimfh, my ($rim_info_packed),8;
    my ($res_count,$keyoffset)=unpack('V2',$rim_info_packed);
    for (my $res_id=0; $res_id<$res_count; $res_id++){
        sysseek $rimfh, $keyoffset +  (32 * $res_id), 0;
        sysread $rimfh, my ($key_packed), 32;
        my ($res_name,$res_type,$res_id2,$res_offset,$res_size)=unpack('a16V4',$key_packed);
        $res_name=~s/\W+//g;
        for my $key (keys %inf) {
            if ( ($inf{$key}{'sought'}) && ($res_type==$resource_numbers{$key}) ) {
                sysseek $rimfh, $res_offset, 0;
                sysread $rimfh, my ($res), $res_size;
                if ($key eq 'ncs') {
                    #ncs files are not gff format....
                    $inf{$key}{'code'}->(\$res,$res_name,$rim);
                }
                else {
                    my $gff=Bioware::GFF->new();
                    $gff->read_gff_scalar(\$res);
                    $gff->{'Name'}=$res_name;
                    $gff->{'File'}=$rim;
                    $inf{$key}{'code'}->(\$gff);
                }
                last;
            }
        }
    }
}

sub search_ncs {
    my ($res_ref,$res_name,$rim)=@_;
    if ($$res_ref=~/\b(\w*?$searchterm\w*?)\b/) {
        #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $rim, $res_name\.ncs Value: $1\n";
        print_it($rim,"$res_name\.ncs"," ",$1);
        $oldrim_len=0;
    }
}
sub search_ifo {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(Mod_OnAcquireItem Mod_OnActivateItem Mod_OnClientLeave Mod_OnHeartbeat Mod_OnLoad Mod_OnStart Mod_OnPlrDeath
    Mod_OnPlrLvlUp Mod_OnPlrRest Mod_OnSpawnBtnDn Mod_OnUnAqreItem Mod_OnUsrDefined);
    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value}=~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.ifo",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.ifo $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }
}
sub search_uti {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(Tag);
    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value}=~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.uti",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.uti $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }

}
sub search_are {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(OnEnter OnExit OnHeartbeat OnUserDefined);
    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value} =~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.are",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.are $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }
}
sub search_utc {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(ScriptHeartbeat ScriptAttacked ScriptDamaged ScriptDeath ScriptDialogue ScriptDisturbed ScriptEndDialogu ScriptEndRound
    ScriptOnNotice ScriptRested ScriptSpawn ScriptSpellAt ScriptUserDefine Tag);
    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value} =~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.utc",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.utc $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }
}
sub search_utd {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(OnClick OnClosed OnDamaged OnDeath OnFailToOpen OnHeartbeat OnLock
    OnMeleeAttacked OnOpen OnSpellCastAt OnTrapTriggered OnUnlock OnUserDefined Tag);

    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value} =~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.utd",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.utd $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }
}
sub search_utm {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(OnOpenStore Tag);
    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value} =~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.utm",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.utm $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }
}
sub search_utp {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(OnClosed OnDamaged OnDeath OnDisarm OnEndDialogue OnHeartbeat OnInvDisturbed OnLock OnMeleeAttacked OnOpen OnSpellCastAt
    OnTrapTriggered OnUnlock OnUsed OnUserDefined Tag);
    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value} =~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.utp",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.utp $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }
}
sub search_utt {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my @fields_to_search=qw(ScriptHeartbeat ScriptOnEnter ScriptOnExit ScriptUserDefine Tag);
    for my $f (@fields_to_search) {
        my $ix=$gff->{Main}->get_field_ix_by_label($f);
        next unless defined $ix;
        if ($gff->{Main}{Fields}[$ix]{Value} =~/$searchterm/) {
            print_it($gff->{File},"$gff->{Name}\.utt",$f,$gff->{Main}{Fields}[$ix]{Value});
            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.utt $f  Value:'$gff->{Main}{Fields}[$ix]{Value}'\n";
            $oldrim_len=0;
        }
    }
}
sub search_dlg {
    my $gff_ref=shift;
    my $gff=$$gff_ref;
    my $slix=$gff->{Main}->get_field_ix_by_label('StartingList');
    my $entix=$gff->{Main}->get_field_ix_by_label('EntryList');
    my $repix=$gff->{Main}->get_field_ix_by_label('ReplyList');
    if (defined $slix) {
        my @startinglist_structs=@{$gff->{Main}{Fields}[$slix]{Value}};
        for my $startinglist_struct (@startinglist_structs) {
            my $activeix=$startinglist_struct->get_field_ix_by_label('Active');
            if (defined $activeix) {
                if ($startinglist_struct->{Fields}[$activeix]{Value} =~/$searchterm/) {
                    print_it($gff->{File},"$gff->{Name}\.dlg","Starting $startinglist_struct->{ID}",$startinglist_struct->{Fields}[$activeix]{Value});
                    #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.dlg StartingList $startinglist_struct->{ID} Value:'$startinglist_struct->{Fields}[$activeix]{Value}'\n";
                    $oldrim_len=0;
                }
            }
        }
    }
    if (defined $entix) {
        my @entrylist_structs=@{$gff->{Main}{Fields}[$entix]{Value}};
        for my $entrylist_struct (@entrylist_structs) {
            my $script_ix=$entrylist_struct->get_field_ix_by_label('Script');
            if (defined $script_ix) {
                if ($entrylist_struct->{Fields}[$script_ix]{Value} =~/$searchterm/) {
                    print_it($gff->{File},"$gff->{Name}\.dlg","Entry $entrylist_struct->{ID}",$entrylist_struct->{Fields}[$script_ix]{Value});
                #    unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.dlg EntryList $entrylist_struct->{ID}  Value:'$entrylist_struct->{Fields}[$script_ix]{Value}'\n";
                    $oldrim_len=0;
                }
            }
            my $replieslist_ix=$entrylist_struct->get_field_ix_by_label('RepliesList');
            if (defined $replieslist_ix) {
                my @replieslist_structs=@{$entrylist_struct->{Fields}[$replieslist_ix]{Value}};
                for my $replieslist_struct (@replieslist_structs) {
                    my $active_ix=$replieslist_struct->get_field_ix_by_label('Active');
                    if (defined $active_ix) {
                        if ($replieslist_struct->{Fields}[$active_ix]{Value} =~/$searchterm/) {
                            print_it($gff->{File},"$gff->{Name}\.dlg","Entry $entrylist_struct->{ID} Replies $replieslist_struct->{ID}",$replieslist_struct->{Fields}[$active_ix]{Value});
                         #   unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.dlg EntryList $entrylist_struct->{ID} RepliesList $replieslist_struct->{ID}  Value:'$replieslist_struct->{Fields}[$active_ix]{Value}'\n";
                            $oldrim_len=0;
                        }
                    }
                }
            }
        }
    }
    if (defined $repix) {
        my @replylist_structs=@{$gff->{Main}{Fields}[$repix]{Value}};
        for my $replylist_struct (@replylist_structs) {
            my $script_ix=$replylist_struct->get_field_ix_by_label('Script');
            if (defined $script_ix) {
                if ($replylist_struct->{Fields}[$script_ix]{Value} =~/$searchterm/) {
                    print_it($gff->{File},"$gff->{Name}\.dlg","Reply $replylist_struct->{ID}",$replylist_struct->{Fields}[$script_ix]{Value});
                    #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.dlg ReplyList $replylist_struct->{ID}  Value:'$replylist_struct->{Fields}[$script_ix]{Value}'\n";
                    $oldrim_len=0;
                }
            }
            my $entrieslist_ix=$replylist_struct->get_field_ix_by_label('EntriesList');
            if (defined $entrieslist_ix) {
                my @entrieslist_structs=@{$replylist_struct->{Fields}[$entrieslist_ix]{Value}};
                for my $entrieslist_struct (@entrieslist_structs) {
                    my $active_ix=$entrieslist_struct->get_field_ix_by_label('Active');
                    if (defined $active_ix) {
                        if ($entrieslist_struct->{Fields}[$active_ix]{Value} =~/$searchterm/) {
                            print_it($gff->{File},"$gff->{Name}\.dlg","Reply $replylist_struct->{ID} Entries $entrieslist_struct->{ID}",$entrieslist_struct->{Fields}[$active_ix]{Value});
                            #unless ($opt_quiet) {print "\n"}; print "Found $scriptref in $gff->{File}, $gff->{Name}\.dlg ReplyList $replylist_struct->{ID} EntriesList $entrieslist_struct->{ID}  Value:'$entrieslist_struct->{Fields}[$active_ix]{Value}'\n";
                            $oldrim_len=0;
                        }
                    }
                }
            }
        }
    }
}

sub search_bif23 { #search templates.bif
    {  #this bit adds speed... it previews the bif and skips it if the string isn't found
     (open my ($biffh),"<","data/templates.bif") or (die "Couldn't open templates.bif $!");
     local $/=\32768;                             # go into slurp mode (chunk)
     my $foundit=0;
     my $bif_chunk;
     my $last_chunk;
     while (my $this_chunk=<$biffh>) {             # slurp chunks
        $bif_chunk=$this_chunk.$last_chunk;        # just in case string is split by chunk boundary
        if ($bif_chunk=~/$scriptref/) {
            $foundit=1;
            last;
        }
        $last_chunk=$this_chunk;
     }
     return unless ($foundit);     # take a peek
    }


    my $keyfh;
    (open $keyfh,'<','chitin.key') or die ("couldn't open chitin.key! $!");
    sysseek $keyfh,12,0;
    sysread $keyfh, my ($keycount_packed),4;
    my $keycount=unpack('V',$keycount_packed);

    sysseek $keyfh,20,0;
    sysread $keyfh,my ($keyentry_offset_packed),4;
    my $keyentry_offset=unpack('V',$keyentry_offset_packed);
    my %bif_entries;
    for (my $i=0; $i<$keycount; $i++) {
        sysseek $keyfh,$keyentry_offset+($i*22),0;
        sysread $keyfh,my ($keyentry_info_packed),22;
        my ($res,$restype,$resid)=unpack('a16vV',$keyentry_info_packed);
        $res=~s/\W+//g;
        for my $key (keys %inf) {
            if ( ($inf{$key}{'sought'}) && ($restype==$resource_numbers{$key}) ) {
                push @{$bif_entries{$resid >> 20}{'Entry'}}, ($resid - (($resid >> 20) << 20));
                push @{$bif_entries{$resid >> 20}{'ResRef'}}, "$res";
                push @{$bif_entries{$resid >> 20}{'code'}},$inf{$key}{'code'};
            }
        }
    }
    sysseek $keyfh,16,0;
    chdir ("data");
    (open my ($biffh),"<","templates.bif") or (die "Couldn't open templates.bif $!");


    for (my $i=0;$i<scalar @{$bif_entries{23}{'Entry'}}; $i++) {
        my $ix=$bif_entries{23}{'Entry'}[$i];
        my $resref=$bif_entries{23}{'ResRef'}[$i];
        sysseek $biffh, 20+(16*$ix),0;
        sysread $biffh, my ($res_info_packed),16;
        my (undef, $offset, $size, undef)=unpack('V4',$res_info_packed);
        sysseek $biffh, $offset, 0;
        sysread $biffh, my ($res), $size;
        my $gff=Bioware::GFF->new();
        $gff->read_gff_scalar(\$res);
        $gff->{'Name'}=$resref;
        $gff->{'File'}='templates.bif';
        $bif_entries{23}{'code'}[$i]->(\$gff);
    }
    close $biffh;
}
sub search_bif21 { #search scripts.bif  (no slurping necessary cuz this file is small and fast already)
    my $keyfh;
    (open $keyfh,'<','chitin.key') or (die "couldn't open chitin.key! $!");
    sysseek $keyfh,12,0;
    sysread $keyfh, my ($keycount_packed),4;
    my $keycount=unpack('V',$keycount_packed);

    sysseek $keyfh,20,0;
    sysread $keyfh,my ($keyentry_offset_packed),4;
    my $keyentry_offset=unpack('V',$keyentry_offset_packed);
    my %bif_entries;
    for (my $i=0; $i<$keycount; $i++) {
        sysseek $keyfh,$keyentry_offset+($i*22),0;
        sysread $keyfh,my ($keyentry_info_packed),22;
        my ($res,$restype,$resid)=unpack('a16vV',$keyentry_info_packed);
        $res=~s/\W+//g;
        next unless $restype == $resource_numbers{'ncs'};
        push @{$bif_entries{$resid >> 20}{'Entry'}}, ($resid - (($resid >> 20) << 20));
        push @{$bif_entries{$resid >> 20}{'ResRef'}}, "$res";
    }
    sysseek $keyfh,16,0;
    chdir ("data");
    (open my ($biffh),"<","scripts.bif") or (die "Couldn't open scripts.bif $!");
    for (my $i=0;$i<scalar @{$bif_entries{21}{'Entry'}}; $i++) {
        my $ix=$bif_entries{21}{'Entry'}[$i];
        my $resref=$bif_entries{21}{'ResRef'}[$i];
        sysseek $biffh, 20+(16*$ix),0;
        sysread $biffh, my ($res_info_packed),16;
        my (undef, $offset, $size, undef)=unpack('V4',$res_info_packed);
        sysseek $biffh, $offset, 0;
        sysread $biffh, my ($res), $size;
        search_ncs(\$res,$resref,'scripts.bif');
    }
    close $biffh;
}
sub print_it {
    my ($filename,$resref,$aux,$found)=@_;
    printf("%-16s%-21s%-19s '%s'\n",$filename,$resref,$aux,$found);
}