#! perl
#
###########################################################
# kmm.pl version 0.2b 
# Copyright (C) 2004 Chuck Chargin Jr. (cchargin@comcast.net)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
##########################################################
# History:
#	
# July 10 2004: First public release of kmm.pl version 0.1
#                
# July 20 2004: version 0.2
#               Added .uts to mod extensions (thanks Darth333)
#               Changed 'system del' to unlink (thanks tk102)
#               Changed list boxes to trees
#               Added mod groups
#               Added 'deactivate all' button
#
# August 25 2004: version 0.2a
#                 Added .lip to mod extensions (thanks RedHawke)
#                 Fixed a bug that could leave a file destination empty
#
# October 18 2004:version 0.2b
#                 Added .jrl to mod extensions (thanks RedHawke)
#                 Added .rim to mod extensions (thanks Shangoman)
#               
##########################################################
#
# This is a script to manage kotor mods.
#
# features:
# -keeps track of all your installed mods
# -when a mod is activated it copies the files to override
# -when a mod is de-activated it removes the files from override
# -supports mods that put files in places other than override
# -checks for conflicts between mods
# -mod groups let you activate a bunch of mods with 1 click
#
# usage:
# perl script: perl perl.pl
#
# compiled script: double click kmm.exe
#
# see readme for more info and ditribution information

use strict;
use Tk;
use Tk::Tree;
use File::Find;
use Win32::FileOp;
use Win32::GUI;
use Win32::TieRegistry (Delimiter => "/");
use Cwd;

use vars qw($VERSION);
$VERSION = '0.2b';

#read the location of kotor from the registry
our $kotordir = $Registry->{"LMachine/Software/bioware/sw/kotor/path"} . "\\";

our @activemods;     # all of the currently active mods
our @inactivemods;   # all of the currently inactive mods
our %modslist;       # all the mods and their info
our $workingdir;     # where KMM is running from
our $moddir;         # where the user says we should find the mods
our %modlookup;      # has mod names as keys and the index as values
our @modindex;       # index corresponds to the value in %modlookup, value is the mod name
our %modgroups;      # holds the mod groups

our $texteditor = "notepad";  # the default text editor

# if the file extension is not in this list then kmm will not touch it 
our %modextensions = ('.uti' => 1, '.tga' => 1, '.2da' => 1, '.mdl' => 1, '.mdx' => 1, '.mod' => 1, '.utc' => 1, '.utp' => 1, '.txi' => 1, '.lyt' => 1, '.dlg' => 1, '.ncs' => 1, '.wav' => 1, '.utm' => 1, '.itp' => 1, '.uts' => 1, '.lip' => 1, '.rim' => 1, '.jrl' => 1);

our %object;
$object{'main'} = MainWindow->new;
$object{'main'}->configure(-title => "Kotor Mod Manager (ver: $VERSION)");

our ($maxwidth, $maxheight) = $object{'main'}->maxsize;

$object{'inactivelist'} = $object{'main'}->Scrolled('Tree',-itemtype => 'text',
                                                     -browsecmd => \&inactiveclick,
						     -width => 55,
						     -height => 18,
						     -selectbackground => 'darkblue',
						     -selectforeground => 'white',
                                                     -scrollbars => 'e');

$object{'activelist'} = $object{'main'}->Scrolled('Tree',-itemtype => 'text',
                                                     -browsecmd => \&activeclick,
						     -width => 55,
						     -height => 18,
						     -selectbackground => 'darkblue',
						     -selectforeground => 'white',
                                                     -scrollbars => 'e');
					     
$object{'inactivetext'} = $object{'main'}->Label(-text => 'Inactive mods:');
$object{'activetext'} = $object{'main'}->Label(-text => 'Active mods:');
$object{'addmodbutton'} = $object{'main'}->Button(-text=>"Add mod", -command => \&addmod);
$object{'delmodbutton'} = $object{'main'}->Button(-text=>"Delete mod", -command => \&deletemod);
$object{'activatemodbutton'} = $object{'main'}->Button(-text=>"Activate mod", -command => \&activatemod);
$object{'deactivatemodbutton'} = $object{'main'}->Button(-text=>"Deactivate mod", -command => \&deactivatemod);
$object{'deactivateallbutton'} = $object{'main'}->Button(-text=>"Deactivate ALL", -command => [\&deactivatemod, "all"]);
$object{'describebutton'} = $object{'main'}->Button(-text=>"Mod description", -command => \&describemod);
$object{'creategroupbutton'} = $object{'main'}->Button(-text=>"Create group", -command => \&creategroup);

$object{'inactivetext'}->form(-t => '%0',
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'inactivelist'}->form(-t => [$object{'inactivetext'},0],
	                 -tp => 10,
			 -l => '%0',
			 -lp => 10,
			 -b => [$object{'activetext'},0],
			 -bp => 10,
		         -r => [$object{'describebutton'},0]);

$object{'activetext'}->form(-t => '%45',
	                    -tp => 10,
			    -l => '%0',
			    -lp => 10);
		      
$object{'activelist'}->form(-t => [$object{'activetext'},0],
	                 -tp => 10,
			 -l => '%0',
			 -lp => 10,
			 -b => '%100',
			 -bp => 10,
		         -r => [$object{'describebutton'},0]);

$object{'activatemodbutton'}->form(-l => [$object{'inactivelist'},0],
	                           -lp => 10,
			           -t => '%0',
			           -tp => 10);

$object{'deactivatemodbutton'}->form(-l => [$object{'inactivelist'},0],
	                             -lp => 10,
			             -t => [$object{'activatemodbutton'},0],
			             -tp => 10);

$object{'deactivateallbutton'}->form(-l => [$object{'inactivelist'},0],
	                             -lp => 10,
			             -t => '%25',
			             -tp => 10);

$object{'describebutton'}->form(-r => '%100',
	                        -lp => 10,
			        -t => '%45',
			        -rp => 10);

$object{'creategroupbutton'}->form(-r => '%100',
	                           -rp => 10,
				   -t => [$object{'describebutton'},0],
				   -tp => 10);

$object{'addmodbutton'}->form(-l => [$object{'activelist'},0],
	                      -lp => 10,
		              -b => [$object{'delmodbutton'},0],
		              -bp => 10);

$object{'delmodbutton'}->form(-l => [$object{'activelist'},0],
	                      -lp => 10,
		              -b => '%100',
		              -bp => 10);

$object{'main'}->update;
$object{'main'}->geometry('+' . int($maxwidth/2 - $object{'main'}->width/2) . '+' . int($maxheight/2 - $object{'main'}->height/2) ); 

# create set prefs window
$object{'setprefs'} = $object{'main'}->Toplevel('title' => 'KMM: Set preferences');
$object{'kotordirentry'} = $object{'setprefs'}->Entry(-width => 45);
$object{'moddirentry'} = $object{'setprefs'}->Entry(-width => 45);
$object{'kotordirbutton'} = $object{'setprefs'}->Button(-text=>"Kotor directory:", 
	                                                -command => [\&getfolder, 'kotordirentry', ""]);
$object{'moddirbutton'} = $object{'setprefs'}->Button(-text=>"Mod directory:", 
	                                              -command => [\&getfolder, 'moddirentry', ""]);
$object{'setprefsbutton'} = $object{'setprefs'}->Button(-text=>"OK", -command => \&setprefs);
$object{'setprefs'}->protocol('WM_DELETE_WINDOW', [\&withdrawwindow, 'setprefs']);
$object{'setprefs'}->withdraw;

$object{'kotordirbutton'}->form(-t => '%0',
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'kotordirentry'}->form(-t => '%0',
	                       -tp => 10,
			       -l => [$object{'kotordirbutton'},0],
			       -lp => 10,
			       -r => '%100',
			       -rp => 10);
		       
$object{'moddirbutton'}->form(-t => [$object{'kotordirbutton'},0],
	                      -tp => 10,
			      -l => '%0',
			      -lp => 10);

$object{'moddirentry'}->form(-t => [$object{'kotordirentry'},0],
	                       -tp => 10,
			       -l => [$object{'moddirbutton'},0],
			       -lp => 10,
			       -r => '%100',
			       -rp => 10);
		       
$object{'setprefsbutton'}->form(-l => '%45',
	                        -t => [$object{'moddirbutton'},0],
				-tp => 10);

$object{'setprefs'}->update;
$object{'setprefs'}->geometry('+' . int($maxwidth/2 - $object{'setprefs'}->width/2) . '+' . int($maxheight/2 - $object{'setprefs'}->height/2) ); 

# create data entry window
$object{'dataentrywin'} = $object{'main'}->Toplevel('title' => 'KMM: Data entry');
$object{'datatext'} = $object{'dataentrywin'}->Label(-text => "Group name:", -width => 12);
$object{'dataentry'} = $object{'dataentrywin'}->Entry(-width => 45);
$object{'commitdatabutton'} = $object{'dataentrywin'}->Button(-text=>"OK", -command => \&commitdata);
$object{'dataentrywin'}->protocol('WM_DELETE_WINDOW', [\&withdrawwindow, 'dataentrywin']);
$object{'dataentrywin'}->withdraw;

$object{'datatext'}->form(-l => '%0',
	                  -lp => 10,
			  -t => '%0',
			  -tp => 10);

$object{'dataentry'}->form(-l => [$object{'datatext'},0],
	                   -lp => 10,
			   -t => '%0',
			   -tp => 10,
			   -r => '%100',
			   -rp => 10);

$object{'commitdatabutton'}->form(-l => '%45',
	                          -t => [$object{'dataentry'},0],
				  -tp => 10);

$object{'dataentrywin'}->update;
$object{'dataentrywin'}->geometry('+' . int($maxwidth/2 - $object{'dataentrywin'}->width/2) . '+' . int($maxheight/2 - $object{'dataentrywin'}->height/2) ); 
			
# create add mod window
$object{'addmodwin'} = $object{'main'}->Toplevel('title' => 'Add mod');
$object{'addmodwin'}->protocol('WM_DELETE_WINDOW', [\&withdrawwindow, 'addmodwin']);
$object{'addmodwin'}->withdraw;
$object{'modnametext'} = $object{'addmodwin'}->Label(-text => "Mod name:", -width => 12);
$object{'modnameentry'} = $object{'addmodwin'}->Entry();
$object{'modauthortext'} = $object{'addmodwin'}->Label(-text => "Author:", -width => 12);
$object{'modauthorentry'} = $object{'addmodwin'}->Entry();
$object{'modvertext'} = $object{'addmodwin'}->Label(-text => "Mod version:", -width => 12);
$object{'modverentry'} = $object{'addmodwin'}->Entry();
$object{'moddescfiletext'} = $object{'addmodwin'}->Label(-text => "Description file:", -width => 12);
$object{'moddescfileentry'} = $object{'addmodwin'}->Entry();
$object{'modpathbutton'} = $object{'addmodwin'}->Button(-text=>"Choose dir", -command => \&addmod);
$object{'modpathentry'} = $object{'addmodwin'}->Entry(-width => 55);
$object{'hiddenentry'} = $object{'addmodwin'}->Entry();
$object{'filelist'} = $object{'addmodwin'}->Scrolled('Listbox', 
                                             -scrollbars => 'se',
			                     -width => 75,
			                     -height => 12);
$object{'commitbutton'} = $object{'addmodwin'}->Button(-text=>"Add the mod", -command => [\&commitaddmod, 'add']);
$object{'modsupbutton'} = $object{'addmodwin'}->Button(-text=>"MOD/SUP", -command => \&modsuptoggle);
$object{'updatekmmbutton'} = $object{'addmodwin'}->Button(-text=>"Add and update kmm", -command => [\&commitaddmod, 'update']);
$object{'destinationbutton'} = $object{'addmodwin'}->Button(-text=>"Destination", -command => \&destination);

$object{'modsupbutton'}-> form(-t => '%35',
	                       -r => '%100',
			       -rp => 10);

$object{'destinationbutton'}->form(-t => [$object{'modsupbutton'},0],
	                           -tp => 10,
				   -r => '%100',
				   -rp => 10);
		       
$object{'modnametext'}->form(-t => '%0',
	                     -tp => 10,
			     -l => '%0',
			     -lp => 10);

$object{'modnameentry'}->form(-t => '%0',
	                      -tp => 10,
			      -l => [$object{'modnametext'},0],
			      -lp => 10,
  		               -r => [$object{'modsupbutton'},0],
		               -rp => 10);

$object{'modauthortext'}->form(-t => [$object{'modnametext'},0],
	                     -tp => 10,
			     -l => '%0',
			     -lp => 10);

$object{'modauthorentry'}->form(-t => [$object{'modnameentry'},0],
	                      -tp => 10,
			      -l => [$object{'modauthortext'},0],
			      -lp => 10,
  		               -r => [$object{'modsupbutton'},0],
		               -rp => 10);

$object{'modvertext'}->form(-t => [$object{'modauthortext'},0],
	                     -tp => 10,
			     -l => '%0',
			     -lp => 10);

$object{'modverentry'}->form(-t => [$object{'modauthorentry'},0],
	                      -tp => 10,
			      -l => [$object{'modvertext'},0],
			      -lp => 10,
  		               -r => [$object{'modsupbutton'},0],
		               -rp => 10);

$object{'moddescfiletext'}->form(-t => [$object{'modvertext'},0],
	                       -tp => 10,
			       -l => '%0',
			       -lp => 10);

$object{'moddescfileentry'}->form(-t => [$object{'modverentry'},0],
	                      -tp => 10,
			      -l => [$object{'moddescfiletext'},0],
			      -lp => 10,
		              -r => [$object{'modsupbutton'},0],
		              -rp => 10);
		      
$object{'modpathbutton'}->form(-t => [$object{'moddescfiletext'},0],
	                       -tp => 10,
			       -l => '%0',
			       -lp => 10,
		               -bp => 10);

$object{'modpathentry'}->form(-t => [$object{'moddescfileentry'},0],
	                      -tp => 10,
			      -l => [$object{'modpathbutton'},0],
			      -lp => 10,
		              -r => [$object{'modsupbutton'},0],
		              -rp => 10);

$object{'filelist'}->form(-t => [$object{'modpathentry'},0],
	                   -tp => 10,
			   -l => '%0',
			   -lp => 10,
			   -r => [$object{'modsupbutton'},0],
			   -rp => 10,
			   -b => [$object{'commitbutton'},0],
			   -bp => 10);
			   
$object{'commitbutton'}->form(-l => '%35',
	                      -b => '%100',
			      -bp => 10,
			      -tp => 10);
		      
$object{'updatekmmbutton'}->form(-l => 	[$object{'commitbutton'},0],
                                 -lp => 10,
			         -b => '%100',
			         -bp => 10);

$object{'addmodwin'}->update;
$object{'addmodwin'}->geometry('+' . int($maxwidth/2 - $object{'addmodwin'}->width/2) . '+' . int($maxheight/2 - $object{'addmodwin'}->height/2) ); 

# initialize the stuff
&kmminit;

# start the freakin' program already!
MainLoop;

sub kmminit {
  my $temp;
  my %grouplookup;
	
  $workingdir = getcwd;
  $workingdir =~ s#/#\\#g;
  # load the configuration file or make one
  if (-e $workingdir . "\\kmm.conf") {
    open(KMMCONFIN, $workingdir . "\\kmm.conf") or die "can't open installedmods.kmm\n";
    while (<KMMCONFIN>) {
      if(/<moddir>=(.*)/) {
        $moddir = $1;
      } elsif (/<kotordir>=(.*)/) {
        $kotordir = $1;
      }
    }
    close KMMCONFIN;
  } else { # no configiration file found, so open up a dialog and get the info from the user
    $object{'kotordirentry'}->delete('0', 'end');
    $object{'kotordirentry'}->insert('end', $kotordir);
  
    $object{'moddirentry'}->delete('0', 'end');
    $object{'moddirentry'}->insert('end', $workingdir);

    $object{'main'}->iconify;
    $object{'setprefs'}->deiconify;
    $object{'setprefs'}->raise;
    $object{'setprefs'}->focus;
  }

  # create the tree roots
  $object{'inactivelist'}->add('.', 
	                     -text => "Inactive mods",
                             -data => "inactive");

  $object{'activelist'}->add('.', 
	                   -text => "Active mods",
                           -data => "active");

  # load the mods list
  if (-e $workingdir . "\\installedmods.kmm") {
    open(IMKMMIN, $workingdir . "\\installedmods.kmm") or die "can't open installedmods.kmm\n";
    while (<IMKMMIN>) {
      if (/(.*\\)(.*\....),(.*)/) {
        if (-e $1.$2) {
          &readkmm($1 . $2, $1, 'init');
  	  $temp = $#modindex;
          $modslist{$temp}{'status'} = $3;
        }
      }
    }
    close IMKMMIN;
  }

  # load the groups list
  if (-e $workingdir . "\\modgroups.kmm") {
    open(kmmgroupin, $workingdir . "\\modgroups.kmm") or die "can't open modgroups.kmm\n";
    $temp = -1;
    while (<kmmgroupin>) {
      if (/<group>=(.*)/) {
        $temp++;
	$grouplookup{$1} = $temp;
        $modgroups{$temp} = {};
        $modgroups{$temp}{'name'} = $1;
        $modgroups{$temp}{'status'} = 'inactive';
        $modgroups{$temp}{'mods'} = {};
      } elsif (/<mod>=(.*)/) {
        $modgroups{$temp}{'mods'}{$modlookup{$1}} = $1;
      }
    }
    close kmmgroupin;
  }

  # figure out what is active
  foreach ( keys %modslist ) {
    if ($modslist{$_}{'status'} eq "active") {
      push @activemods, $_;
    } elsif ($modslist{$_}{'status'} eq "inactive" ) {
      push @inactivemods, $_;
    } else {
      $modgroups{$grouplookup{$modslist{$_}{'status'}}}{'status'} = 'active';
    }
  }
  foreach (keys %modgroups) {
    if ($modgroups{$_}{'status'} eq 'active') {
      push @activemods, 'G_' . $_;
    } else {
      push @inactivemods, 'G_' . $_;
    }
  }
  
  &listfill('both');
}

sub writegroups {
  my $group;
  my $mod;
	
  open(KMMGROUPOUT, ">", $workingdir . "\\modgroups.kmm") or die "can't open modgroups.kmm\n";

  foreach $group ( keys %modgroups ) {
    
    print(KMMGROUPOUT "<group>=" . $modgroups{$group}{'name'} . "\n");
    foreach $mod ( keys %{ $modgroups{$group}{'mods'} } ) {
      print(KMMGROUPOUT "<mod>=" . $modindex[$mod] . "\n");
    }
    print(KMMGROUPOUT "<endgroup>". $modgroups{$group}{'name'} . "\n");
  }
  close KMMGROUPOUT;
}

sub commitdata {
  my ($temp, $target);

  $target = $object{'dataentry'}->get;

  foreach (keys %modgroups) {
    if ($modgroups{$_}{'name'} eq $target) {
      return -1;
    }
  }
  
  $temp = 0;
  while ( defined( $modgroups{$temp} ) ) {
    $temp++;
  }
 
  $modgroups{$temp} = {};
  $modgroups{$temp}{'name'} = $target;
  $modgroups{$temp}{'mods'} = {};
  foreach ( @activemods ) {
    $modgroups{$temp}{'mods'}{$_} = $modindex[$_];
    $modslist{$_}{'status'} = $modgroups{$temp}{'name'};
  }

  &writegroups;
  
  @activemods = ();
  push @activemods, "G_" . $temp;

  &writekmmim;
  
  $object{'dataentrywin'}->withdraw;
  &listfill('active');
}

sub creategroup {
  my $index;
	
  $index = $object{'activelist'}->info('children', '.');
  if ($index eq "") {return;}
  
  foreach (@activemods) {
    if (/G_/) {return;}
  }
  
  $object{'dataentry'}->delete('0', 'end');
  $object{'dataentrywin'}->deiconify;
  $object{'dataentrywin'}->raise;
  $object{'dataentrywin'}->focus;
}

sub inactiveclick {
  $object{'activelist'}->selectionClear;
}

sub activeclick {
  $object{'inactivelist'}->selectionClear;
}

sub setprefs {
  $kotordir = $object{'kotordirentry'}->get;
  $moddir = $object{'moddirentry'}->get;
  open(KMMCONFOUT, '>', $workingdir . "\\kmm.conf") or die "can't open installedmods.kmm\n";
  print(KMMCONFOUT "<moddir>=" . $moddir . "\n");
  print(KMMCONFOUT "<kotordir>=" . $kotordir . "\n");
  close KMMCONFOUT;
  $object{'setprefs'}->withdraw;
  $object{'main'}->deiconify;
  $object{'main'}->raise;
  $object{'main'}->focus;
}

# change the destination of a mod file
sub destination {
  my ($index, $type, $target);
  my $temp;

  $object{'hiddenentry'}->delete('0', 'end');
  
  $index = $object{'filelist'}->curselection;
  if ($index ne "") {
    $target = $object{'filelist'}->get($index);
    $type = substr($target, 0, 3);
    if ($target =~ /(.*) <dest>=(.*)/) {
      $target = $1;
    }
    if ($type eq "MOD") {
      $object{'filelist'}->delete($index);
      &getfolder('hiddenentry', $kotordir);
      $temp = $object{'hiddenentry'}->get;
      print ($temp . "\n");
      if ($temp eq "\\") {
        $temp = $kotordir . "override\\";
      }
      $temp = substr($temp, length($kotordir));
      $object{'filelist'}->insert($index, $target . " <dest>=" . $temp);
    } 
  }
}

# toggle if a file is a mod file or a support file
sub modsuptoggle {
  my ($index, $type, $target);
  
  $index = $object{'filelist'}->curselection;
  if ($index ne "") {
    $target = $object{'filelist'}->get($index);
    $object{'filelist'}->delete($index);
    $type = substr($target, 0, 3);
    if ($type eq "MOD") {
      $target =~ /MOD: (.*) <dest>=.*/;
      $object{'filelist'}->insert($index, "SUP: " . $1);
    } else {
      $object{'filelist'}->insert($index, "MOD" . substr($target, 3) . " <dest>=override\\");
    }
  }
}

# open up notepad and display the mod info (if it has a readme file)
sub describemod {
  my ($target, $index);
	
  $index = $object{'activelist'}->selectionGet;
  if ($index ne "") {
    $target = $object{'activelist'}->info('data', $index);
    if ( $modslist{$target}{'description'} ne "" ) {
      system $texteditor . " " . chr(34) . $modslist{$target}{'modroot'} . $modslist{$target}{'description'} . chr(34);
    }
    return;
  }
  
  $index = $object{'inactivelist'}->selectionGet;
  if ($index ne "") {
    $target = $object{'inactivelist'}->info('data', $index);
    if ( $modslist{$target}{'description'} ne "" ) {
      system $texteditor . " " . chr(34) . $modslist{$target}{'modroot'} . $modslist{$target}{'description'} . chr(34);
    }
    return;
  }
}

# remove the mod from KMM
# if the mod is currently installed, then de-install it first
sub deletemod {
  my ($index, $target, $work, $work2);
  my @temp;

  $index = $object{'activelist'}->selectionGet;
  if ($index ne "") {
    # get the selected item
    $target = $object{'activelist'}->info('data', $index);
    # deactivate the mod
    &deactivatemod;
    # select the inactive mod
    $object{'inactivelist'}->selectionSet("." . $target);
    # now that the mod is selected the next block will delete it
  }

  $index = $object{'inactivelist'}->selectionGet;
  if ($index ne "") {
    # get the selected item and parse it
    $target = $object{'inactivelist'}->info('data', $index);
    if ($target =~ /G_(.*)/) {
      delete $modgroups{$1};
      &writegroups;
    } else {
      # delete the mod from all groups
      foreach $work (keys %modgroups) {
        foreach ( keys %{$modgroups{$work}{'mods'}} ) {
	  if ($target == $_) {
	    delete $modgroups{$work}{'mods'}{$_};
	  }
        }
      }
      &writegroups;
      # delete from the 'installedmods.kmm' file
      $modslist{$target} = undef;
      delete $modslist{$target};
      &writekmmim;
      # remove the selected mod from the active list
    }
    foreach ( @inactivemods ) {
      if ($_ eq $target) {next;}
      push @temp, $_;
    }
    @inactivemods = @temp;
    # update the lists  
    &listfill('inactive');
    return;
  } # if ($index ne "")
  
}

# deactivate a mod and remove its files from override (and anywhere else they might be)
sub deactivatemod {
  my ($argument) = (@_);
  my ($index, $target, $modroot, $source, $dest, $file, $rensource, $lcfile, $lctarget);
  my ($Dsize, $Ssize, $Dmodified, $Smodified, @deactivatelist);
  my @temp;
  my @conflictlist;

  # check to see if there is a valid selection in the listbox
  $index = $object{'activelist'}->selectionGet;
  if ($index eq "" && $argument ne "all") {
    return;
  } elsif ($argument ne "all") {
    # get the selected item 
    $target = $object{'activelist'}->info('data', $index);
  }

  # check to see if we are deactivating a single mod or a group, or all
  if ($argument eq "all") {
    foreach $target (@activemods) {
      if ($target =~ /G_(.*)/) {
        push @inactivemods, $target;
        foreach (keys %{$modgroups{$1}{'mods'}} ) {
          push @deactivatelist, $_;
          # add the selected mod to the inactive list
          push @inactivemods, $_;
        }
      } else {
        push @deactivatelist, $target;
        push @inactivemods, $target;
      }
    }
    @activemods = ();
  } elsif ($target =~ /G_(.*)/) { # a group
    push @inactivemods, $target;
    foreach (keys %{$modgroups{$1}{'mods'}} ) {
      push @deactivatelist, $_;
      # add the selected mod to the inactive list
      push @inactivemods, $_;
      # remove the selected mod from the active list
      foreach ( @activemods ) {
        if ($_ eq $target) {next;}
        push @temp, $_;
      }
      @activemods = @temp;
      @temp = ();
    }
  } else { # a single mod
    $deactivatelist[0] = $target;
    # add the selected mod to the inactive list
    push @inactivemods, $target;
    # remove the selected mod from the active list
    foreach ( @activemods ) {
      if ($_ eq $target) {next;}
      push @temp, $_;
    }
    @activemods = @temp;
  }
 
  #check each file and see if it is ours
  foreach $target ( @deactivatelist ) {
    $modroot = $modslist{$target}{'modroot'};
    print("Deactivating: " . $modindex[$target] . "\n");
    foreach $file ( keys %{$modslist{$target}{'modfiles'}} ) {
      $lcfile = lc($file);
      if (! defined( $modextensions{substr($file, length($file)-4)} ) ) {
        print("Skipping illegal file: $file\n");
        next;
      }
      $source = $modroot . $modslist{$target}{'modfiles'}{$lcfile}{'sourcepath'} . $file;
      $dest = $kotordir . $modslist{$target}{'modfiles'}{$lcfile}{'dest'} . $file;
      if (-e $dest) {
        # get the info on the file
        (undef, undef, undef, undef, undef, undef, undef, $Dsize, undef, $Dmodified) = stat($dest);
        # get the info on our file
        (undef, undef, undef, undef, undef, undef, undef, $Ssize, undef, $Smodified) = stat($source);
        if (($Dsize == $Ssize) && ($Dmodified == $Smodified)) {
          #same name, same size, same modified time, must be the same file so delete it
          print("Removing $dest\n");
	  unlink $dest;
  	  # check if this file has a backup
  	  if (-e $kotordir . $modslist{$target}{'modfiles'}{$lcfile}{'dest'} . "_" . $file) {
            $rensource = chr(34) . $kotordir . $modslist{$target}{'modfiles'}{$lcfile}{'dest'} . "_" . $file . chr(34);
            print( "Restoring backed up file: $rensource\n");
            system ("ren " . $rensource . " " . $file);
 	  }
        } else {
          # don't know what we should do with this file, we are not managing it
 	  print("$dest does not belong to $modindex[$target]\n");
  	  push @conflictlist, "$dest does not belong to $modindex[$target]";
	}
      } # if (-e $dest)
    } # foreach $file
    # update the mod status
    $modslist{$target}{'status'} = 'inactive';
  } # foreach $target
  
  # report any file conflicts
  if ( defined($conflictlist[0]) ) {
    print("----------\n");
    foreach (@conflictlist) {
      print("$_\n");
    }
    unshift @conflictlist, "The following files were not removed:\n";
    &showerror(@conflictlist);
  }
  
  # update the 'installedmods.kmm' file
  &writekmmim;

  # update the lists  
  &listfill('both');

}

#activate a mod and copy its files to override and anywhere else they need to go
sub activatemod {
  my ($index, $target, $file, $modroot, $source, $dest, $rensource);
  my @temp;
  my @conflictlist;
  my ($Dsize, $Dmodified, $Ssize, $Smodified);
  my ($foundconflict, $lcfile, $lctarget, $groupname);
  my @activatelist;
  my @activecheck;

  # check to see if there is a valid selection in the listbox
  $index = $object{'inactivelist'}->selectionGet;
  if ($index eq "") {return;}

  # get the selected item
  $target = $object{'inactivelist'}->info('data', $index);

  # figure out if we are activating a group or a single mod
  if ($target =~ /G_(.*)/) { # a group
    $groupname = $modgroups{$1}{'name'};
    foreach (keys %{$modgroups{$1}{'mods'}} ) {
      push @activatelist, $_;
    }
  } else { # single mod
    $groupname = "";
    $activatelist[0] = $target;
  }

  # create list of already active mods
  @activecheck = ();
  foreach ( @activemods ) {
    if (/G_(.*)/) { # a group
      foreach (keys %{$modgroups{$1}{'mods'}} ) {
        push @activecheck, $_;
      }
    } else {
      push @activecheck, $_;
    }
  }

  $foundconflict = 0;
  # check to see if the mod(s) we are activating are already active
  foreach $dest ( @activatelist ) {
    foreach ( @activecheck ) {
      if ($dest == $_) {
        $foundconflict = 1;
	push @conflictlist, "$modindex[$dest] is already active";
      }
    }
  }
  
  foreach $target ( @activatelist ) {
    $modroot = $modslist{$target}{'modroot'};
    # check for file conflicts with activated mods and unmanaged files
    foreach $file ( keys %{$modslist{$target}{'modfiles'}} ) {
      $lcfile = lc($file);
      # check against the other active mods
      foreach ( @activecheck ) {
        if ($_ eq $target) {next;} # don't match against the mod we are activating
        if ( defined( $modslist{lc($_)}{'modfiles'}{$lcfile} ) ) {
	  push @conflictlist, "$file conflicts with file in $modindex[$_]";
  	  $foundconflict = 1;
        }
      }
      # check for unmanaged files that may conflict
      $source = $modroot . $modslist{$target}{'modfiles'}{$lcfile}{'sourcepath'} . $file;
      $dest = $kotordir . $modslist{$target}{'modfiles'}{$lcfile}{'dest'} . $file;
      if (-e $dest && $foundconflict == 0) {
        # get the info on the conflicting file
        (undef, undef, undef, undef, undef, undef, undef, $Dsize, undef, $Dmodified) = stat($dest);
        # get the info on our file
        (undef, undef, undef, undef, undef, undef, undef, $Ssize, undef, $Smodified) = stat($source);
        if (($Dsize == $Ssize) && ($Dmodified == $Smodified)) {
          #same name, same size, same modified time, must be the same file
  	  # delete it so we can put it back in the next loop, silly huh?
          print("Removing $dest\n");
  	  unlink $dest;
        } else {
          # don't know what we should do with this file, we are not managing it
  	  push @conflictlist, "$dest already exists.";
        }
      }
      # $foundconflict = 0;
    } # foreach $file
  } # foreach $target
  if ( defined($conflictlist[0]) ) {
    print("----------\n");
    foreach (@conflictlist) {
      print("$_\n");
    }
    &showerror(@conflictlist);
    return;
  }

  $target = $object{'inactivelist'}->info('data', $index);
  # we are good to go, add the group to the active list
  push @activemods, $target;
  # remove the group from the inactive list
  @temp = ();
  foreach ( @inactivemods ) {
    if ($_ eq $target) {next;}
    push @temp, $_;
  }
  @inactivemods = @temp;
  
  # now add each mod to the active list
  foreach $target ( @activatelist ) {
    $modroot = $modslist{$target}{'modroot'};
    #copy the files to their locations
    foreach $file ( keys %{$modslist{$target}{'modfiles'}} ) {
      $lcfile = lc($file);
      print("----------------------\n");
      if (! defined( $modextensions{ lc(substr($file, length($file)-4)) } ) ) {
        print("Skipping illegal file: $file\n");
        next;
      }
      $source = chr(34) . $modroot . $modslist{$target}{'modfiles'}{$lcfile}{'sourcepath'} . $file . chr(34);
      $dest = chr(34) . $kotordir . $modslist{$target}{'modfiles'}{$lcfile}{'dest'} . chr(34);
      if (-e $kotordir . $modslist{$target}{'modfiles'}{$lcfile}{'dest'} . $file) {
        $rensource = chr(34) . $kotordir . $modslist{$target}{'modfiles'}{$lcfile}{'dest'} . $file . chr(34);
        print( $rensource . " exists, making backup\n");
        system ("ren " . $rensource . " _" . $file);
        print( "copy " . $source . " " . $dest . "\n");
        system ("copy " . $source . " " . $dest);
      } else {
        print( "copy " . $source . " " . $dest . "\n");
        system ("copy " . $source . " " . $dest);
      }
    }
    # add the selected mod to the active list
    # push @activemods, $target;
    # remove the selected mod from the inactive list
    @temp = ();
    foreach ( @inactivemods ) {
      if ($_ eq $target) {next;}
      push @temp, $_;
    }
    @inactivemods = @temp;
  
    # update the 'installedmods.kmm' file
    if ($groupname ne "") {
      $modslist{$target}{'status'} = $groupname;
    } else {
      $modslist{$target}{'status'} = 'active';
    }
  } # foreach $target
  &writekmmim;
  
  # update the lists  
  &listfill('both');
}

# write out a kmm file for the mod
sub writekmm {
  my ($type, $filename, $filepath, $ext, $size, $modified, $dest);
  my ($temp, $index);
  my $modname = $object{'modnameentry'}->get;
  my $modroot = $object{'modpathentry'}->get;

  if ( defined($modlookup{$modname}) ) {
    $index = $modlookup{$modname};
  } else {
    $index = 0;
    while (defined( $modslist{$index} ) ) {
      $index++;
    }
    $modindex[$index] = $modname;
    $modlookup{$modname} = $index;
  }

  $modslist{$index} = {};
  $modslist{$index}{'modroot'} = $modroot;
  # create the kmm file
  open(KMMOUT, ">", $modroot . "$modname.kmm") or die "can't open out file\n";
  print("writing: " . $modroot . "$modname.kmm\n");
  # the name of the mod
  print(KMMOUT "<mod>=" . $modname . "\n");
  # the author of the mod
  $modslist{$index}{'author'} = $object{'modauthorentry'}->get;
  print(KMMOUT "<author>=" . $object{'modauthorentry'}->get . "\n");
  # the version number of the mod
  $modslist{$index}{'version'} = $object{'modverentry'}->get;
  print(KMMOUT "<version>=" . $object{'modverentry'}->get . "\n");
  # mod description file, usually: "d:\moddir\readme.txt"
  $modslist{$index}{'description'} = $object{'moddescfileentry'}->get;
  print(KMMOUT "<readme>=" . $object{'moddescfileentry'}->get . "\n");

  foreach (sort $object{'filelist'}->get('0', 'end')) {
    if (/\.kmm/) {next;} # skip kmm files
    # check if the file has a non-override destination
    # if it does chop it off and store it
    if (/(.*) <dest>=(.*)/) {
      $temp = $1;
      $dest = $2;
    } else {
      $dest = 'override\\';
      $temp = $_;
    }
      
    # check if this file is in a subdirectory
    if ( index($temp, '\\') > -1) {
      # the file is in a subdirectory
      $temp =~ /(SUP:|MOD:) (.*\\)(.*)(....)/;
      $type = $1;
      $filepath = $2;
      $filename = $3;
      $ext = $4;
    } else {
      # the file is not in a subdirectory
      $temp =~ /(SUP:|MOD:) (.*)(....)/;
      $type = $1;
      $filepath = "";
      $filename = $2;
      $ext = $3;
    }

    # if this is a mod file add it to the modfiles list
    if ($type eq "MOD:") {
      $modslist{$index}{'modfiles'}{lc($filename . $ext)}{'dest'} = $dest;
      $modslist{$index}{'modfiles'}{lc($filename . $ext)}{'sourcepath'} = $filepath;
      print(KMMOUT "<modfile>=$filepath,$filename$ext,$dest\n");     
    } else {
      # the file is not a mod file, so it must be a support file
      $modslist{$index}{'supportfiles'}{$filename . $ext} = $modroot . $filepath;	
      print(KMMOUT "<supportfile>=$filepath$filename$ext\n");     
    }
  }
  # add an entry for this kmm file we are writing
  print(KMMOUT "<supportfile>=$modname.kmm\n");     

  close KMMOUT;
}

# clear out the add mod window
sub cleanoutaddmod {
  # clear out the entry boxes
  $object{'modpathentry'}->delete('0', 'end');
  $object{'modverentry'}->delete('0', 'end');
  $object{'modauthorentry'}->delete('0', 'end');
  $object{'moddescfileentry'}->delete('0', 'end');
  # clear out the listbox
  $object{'filelist'}->delete('0', 'end');
}

# actually add the mod to the KMM mod list
sub commitaddmod {
  my ($arg) = (@_);
  my ($temp, $index);
  my $modname = $object{'modnameentry'}->get;
  my $root = $object{'modpathentry'}->get;

  # check if there is a kmm file.  If there isn't one create one
  print("looking for " . $root . $modname . ".kmm\n");
  if(! -e $root . $modname . ".kmm") { # no kmm file, so create it
    print("no kmm file, creating one\n");
    &writekmm;
  } elsif ((-e $root . $modname . ".kmm") && ($arg eq 'update')) { #kmm file exists, but an update was requested
    print("kmm file exists, force update requested\n");
    &writekmm;
  } else { # kmm file found, no update requested
    print($root . $modname . ".kmm exists, leaving it alone\n");
  }

  $index = $modlookup{$modname};
  
  # add the mod to the 'installedmods.kmm' file
  $modslist{$index}{'kmmfile'} = $root . $modname . ".kmm";
  $modslist{$index}{'status'} = 'inactive';
  &writekmmim;

  #add the new mod to the inactive list
  push @inactivemods, $index;
  
  # update the inactive list
  &listfill('inactive');
  
  # clear out the add mod window
  &cleanoutaddmod;
  # hide the add mod window
  $object{'addmodwin'}->withdraw;
}

# write the KMM installed mods file
sub writekmmim {
  open(KMMIMOUT, ">", $workingdir . "\\installedmods.kmm") or die "can't open installedmods.kmm\n";
  foreach ( keys %modslist ) {
    print(KMMIMOUT $modslist{$_}{'kmmfile'} . "," . $modslist{$_}{'status'} . "\n");
  }
  close KMMIMOUT;
}

# the sort sub
sub kmmsort {
  if ($a =~ /G_/ && $b =~ /G_/) {
    return $modgroups{substr($a,2)}{'name'} cmp $modgroups{substr($b,2)}{'name'};
  } elsif ($a =~ /G_/) {
    return -1;
  } elsif ($b =~ /G_/) {
    return 1;
  } else {
    return lc($modindex[$a]) cmp lc($modindex[$b]);
  }
}

# fill and sort the active and inactive listboxes
sub listfill {
  my ($tofill) = (@_);
  my ($curmod, $count, $groupnum);

  if ($tofill eq 'both' or $tofill eq 'active') {
    @activemods = sort kmmsort @activemods;
    $object{'activelist'}->delete('all');
    $object{'activelist'}->add('.', 
	                         -text => "active mods",
                                 -data => "active");
    # first we will look at the active mods list
    foreach $curmod ( @activemods ) {
      if ($curmod =~ /G_(.*)/) { # this is an active group
	$groupnum = $1;
	# add the group root
        $object{'activelist'}->add(".G_$curmod", 
                                   -text => "G_" . $modgroups{$groupnum}{'name'},
                                   -data => $curmod);
	# add the mods to the group
        foreach (sort {lc($modindex[$a]) cmp lc($modindex[$b])} keys %{$modgroups{$groupnum}{'mods'}} ) {
          $object{'activelist'}->add(".G_$curmod.$_", 
	                           -text => $modindex[$_],
			           -data => $curmod);
        }
        $object{'activelist'}->setmode(".G_$curmod", "close");
        $object{'activelist'}->close(".G_$curmod");
      } else { # this is a mod not in a group
	# add the mod root, and base info
        $object{'activelist'}->add(".$curmod", 
                                   -text => $modindex[$curmod] . " (ver: " . $modslist{$curmod}{'version'} . ")",
                                   -data => $curmod);
        $object{'activelist'}->add(".$curmod.author", 
  	                         -text => "author: " . $modslist{$curmod}{'author'},
  			         -data => $curmod);
        $object{'activelist'}->add(".$curmod.files", 
  	                         -text => "files",
  			         -data => $curmod);
        $count = 0;
	# add the mod files list
        foreach (sort keys %{$modslist{$curmod}{'modfiles'}} ) {
          $object{'activelist'}->add(".$curmod.files.$count", 
	                           -text => "MOD: " . $_,
			           -data => $curmod);
          $count++;
        }
	# add the support files list
        foreach (sort keys %{$modslist{$curmod}{'supportfiles'}} ) {
          $object{'activelist'}->add(".$curmod.files.$count", 
   	                           -text => "SUP: " . $_,
  			           -data => $curmod);
  	  $count++;
        }
        $object{'activelist'}->setmode(".$curmod.files", "close");
        $object{'activelist'}->close(".$curmod.files");
        $object{'activelist'}->setmode(".$curmod", "close");
        $object{'activelist'}->close(".$curmod");
      } # if $curmod =~ /g(.*)/
    } # foreach $curmod ( @activemods )
  }
  
  # now we look at the inactive list
  if ($tofill eq 'both' or $tofill eq 'inactive') {
    @inactivemods = sort kmmsort @inactivemods;
    $object{'inactivelist'}->delete('all');
    $object{'inactivelist'}->add('.', 
	                         -text => "Inactive mods",
                                 -data => "inactive");
    foreach $curmod ( @inactivemods ) {
      if ($curmod =~ /G_(.*)/) { # this is a group
	$groupnum = $1;
	# create the group root
        $object{'inactivelist'}->add(".G_$curmod", 
                                   -text => "G_" . $modgroups{$groupnum}{'name'},
                                   -data => $curmod);
	# add the mods list to the group
        foreach (sort {lc($modindex[$a]) cmp lc($modindex[$b])} keys %{$modgroups{$groupnum}{'mods'}} ) {
          $object{'inactivelist'}->add(".G_$curmod.$_", 
	                           -text => $modindex[$_],
			           -data => $curmod);
        }
        $object{'inactivelist'}->setmode(".G_$curmod", "close");
        $object{'inactivelist'}->close(".G_$curmod");
      } else { # this is a mod not in a group
	# add the mod root and base info
        $object{'inactivelist'}->add(".$curmod", 
                                   -text => $modindex[$curmod] . " (ver: " . $modslist{$curmod}{'version'} . ")",
                                   -data => $curmod);
        $object{'inactivelist'}->add(".$curmod.author", 
	                           -text => "author: " . $modslist{$curmod}{'author'},
			           -data => $curmod);
        $object{'inactivelist'}->add(".$curmod.files", 
	                           -text => "files",
			           -data => $curmod);
        $count = 0;
	# add the mod files to the mod
        foreach (sort keys %{$modslist{$curmod}{'modfiles'}} ) {
          $object{'inactivelist'}->add(".$curmod.files.$count", 
	                           -text => "MOD: " . $_,
			           -data => $curmod);
	  $count++;
        }
	# add the support files to the mod
        foreach (sort keys %{$modslist{$curmod}{'supportfiles'}} ) {
          $object{'inactivelist'}->add(".$curmod.files.$count", 
	                           -text => "SUP: " . $_,
	  		           -data => $curmod);
	  $count++;
        }
        $object{'inactivelist'}->setmode(".$curmod.files", "close");
        $object{'inactivelist'}->close(".$curmod.files");
        $object{'inactivelist'}->setmode(".$curmod", "close");
        $object{'inactivelist'}->close(".$curmod");
      }
    }
  }
}

# used to find the files when adding a new mod without a KMM file
sub wanted {
  my $root = $object{'modpathentry'}->get;
  my $temp = $File::Find::name;
  my $ext = substr($temp, length($temp)-4);

  if (-f $temp) {
    $temp =~ s#\\/#\\#g;
    $temp =~ s#/#\\#g;
    $temp = substr($temp, length($root));
    if ( defined( $modextensions{lc($ext)} ) ) {
      $object{'filelist'}->insert('end', "MOD: " . $temp . " <dest>=override\\");
    } else {
      $object{'filelist'}->insert('end', "SUP: " . $temp);
    }
  }
}

# start the add mod process
sub addmod {
  my ($temp, $loc, $kmmfile, $readme);
	
  # clear out the add mod window
  &cleanoutaddmod;
  
  #open the file dialog and let the user look for a mod dir
  if (&getfilepath('modpathentry', 'dironly', $moddir) != 1) {return -1;}
    
  # open the selected dir and look for a .kmm file
  my $modroot = $object{'modpathentry'}->get;
  opendir MODDIR, $modroot;
  foreach (readdir MODDIR) {
    if ( /\.kmm/ ) {
      $kmmfile = $modroot . $_;
      last;
    }
  }
  closedir MODDIR;

  # if we found a .kmm file check it out, else collect the mod info the hard way
  if ($kmmfile ne "") {
    print("kmm file '$kmmfile' found\n");
    foreach ( keys %modslist ) {
      if( $modslist{$_}{'kmmfile'} eq $kmmfile ) {
        print("That mod is already installed!\n");
	&showerror("That mod is already installed!");
	return;
      }
    }
    &readkmm($kmmfile, $modroot, 'addmod');
  } else {
    print("no kmm file found\n");
    $temp = $object{'modpathentry'}->get;
    $loc = rindex($temp, "\\", length($temp)-2)+1;
    $temp = substr($temp, $loc, (length($temp)-$loc)-1);
    $object{'modnameentry'}->delete('0', 'end');
    $object{'modnameentry'}->insert('end', $temp);

    $object{'filelist'}->delete('0', 'end');

    # look for a file with 'readme' in the name
    opendir MODDIR, $modroot;
    foreach (readdir MODDIR) {
      if ( /readme/i ) {
        $object{'moddescfileentry'}->delete('0', 'end');
        $object{'moddescfileentry'}->insert('end', $_);
        system "start notepad $modroot$_";
        last;
      }
    }
    closedir MODDIR;
  
    find(\&wanted, $object{'modpathentry'}->get);
  }
  
  $object{'addmodwin'}->deiconify;
  $object{'addmodwin'}->raise;
  $object{'addmodwin'}->focus;
}

# read a KMM file
sub readkmm {
  my ($kmmfile, $modroot, $caller) = (@_);
  my ($modname, $index);

  open(KMMIN, $kmmfile) or die "can't open kmm file\n";
  print("reading $kmmfile\n");
  
  while (<KMMIN>) {
    if (/<mod>=(.*)/) {
      push @modindex, $1;
      $index = $#modindex;
      $modlookup{$1} = $index;
      $modname = $1;
    } elsif (/<author>=(.*)/) {
      $modslist{$index}{'author'} = $1;
    } elsif (/<version>=(.*)/) {
      $modslist{$index}{'version'} = $1;
    } elsif (/<readme>=(.*)/) {
      $modslist{$index}{'description'} = $1;
    } elsif (/<modfile>=(.*),(.*),(.*)/) {
      $modslist{$index}{'modfiles'}{lc($2)}{'sourcepath'} = $1;	
      $modslist{$index}{'modfiles'}{lc($2)}{'dest'} = $3;	
    } elsif (/<supportfile>=(.*)/) {
      $modslist{$index}{'supportfiles'}{$1} = $modroot . $1;	
    }
  }
  close KMMIN;

  $modslist{$index}{'kmmfile'} = $kmmfile;
  #$modslist{$index}{'status'} = ;
  
  $modslist{$index}{'modroot'} = $modroot;
  if ($caller eq "addmod") {
    $object{'modnameentry'}->delete('0', 'end');
    $object{'modnameentry'}->insert('end', $modname);

    $object{'modauthorentry'}->delete('0', 'end');
    $object{'modauthorentry'}->insert('0', $modslist{$index}{'author'});

    $object{'modverentry'}->delete('0', 'end');
    $object{'modverentry'}->insert('0', $modslist{$index}{'version'});

    $object{'moddescfileentry'}->delete('0', 'end');
    $object{'moddescfileentry'}->insert('0', $modslist{$index}{'description'});

    foreach ( sort keys %{$modslist{$index}{'modfiles'}} ) {
      $object{'filelist'}->insert('end', "MOD: " . $modslist{$index}{'modfiles'}{$_}{'sourcepath'} . $_ . " <dest>=" . $modslist{$index}{'modfiles'}{$_}{'dest'});
    }

    foreach ( sort keys %{$modslist{$index}{'supportfiles'}} ) {
      $object{'filelist'}->insert('end', "SUP: " . $_);
    }
  } elsif ($caller eq 'init') {
  }
}

# open a window to select a folder
sub getfolder {
  my ($target, $start) = (@_);

  $object{$target}->delete('0', 'end');
  $object{$target}->insert('end', Win32::GUI::BrowseForFolder(-root => $start) . "\\");
  
}

# open a window to select a file
sub getfilepath {
  my ($entry, $type, $startdir) = (@_);
  my $temp;
  
  my %parms=(title => "Select mod directory", 
	     handle=> 0,
             dir => $startdir,
             filters => { 'All Files' => '*.*'},
             options => OFN_PATHMUSTEXIST);
      
  $temp = OpenDialog \%parms;
  if ($type eq "dironly") {
    $temp =~ /(.*\\)(.*\..*)/;
    $temp = $1;
  }

  if (-e $temp) {
    $object{$entry}->delete('0','end');
    $object{$entry}->insert('end',$temp);
    return 1;
  } else {
    print ("Path/file does not exist!\n");
    return -1;
  }
}

# routine to hide a window instead of destroying it
sub withdrawwindow {
  my ($target) = (@_);
  
  # hide the window
  $object{$target}->withdraw;

}

sub showerror {
  my (@messagelist) = (@_);
  my $message;

  if ( defined($messagelist[1]) ) {
    $message = "The following errors occured:\n\n"; 
  } else {
    $message = "The following error occured:\n\n";
  }

  foreach (@messagelist) {
    $message .= $_ . "\n";
  }
  
  $object{'main'}->messageBox(-message => $message, 
	                      -title => "KMM status", 
			      -type => 'OK');

}

