
#
# DLGEditor
# A GUI dialog editor made for Star Wars: Knights of the Old Republic
#
#version history:
# 0.1 - Sep 5, 2004 - sneak peek - treeview of dlg
# 0.2 - Sep 7, 2004 - added widgets to lower pane showing all info
# 0.3 - Sep 8, 2004 - added View menu and subsequent functions
# 0.4 - Sep 8, 2004 - added Add Entry/Add Reply menu functions
# 0.5 - Sep 10, 2004 - added cut/copy/Paste Node, orphans, refresh_tree_gentle, Tk::Error and lots of cosmetics
# 0.6 - Sep 10, 2004 - added support for opening dialog by argument
# 0.7 - Sep 10, 2004 - changed reference to 'branches' to 'nodes' and added Paste Node as Copy or Paste Node as New to menu
# 0.8 - Sep 11, 2004 - added right click context menus
# 0.9 - Sep 11, 2004 - added Jump To Original to context menu, but dbl-click event handling is hosed so not implemented
# 0.10 - Sep 11, 2004 - added subclassing to Tree widget so that Double Clicking would behave as expected
# 0.11 - Sep 12, 2004 - added keyboard bindings (menu accelerators)
# 0.12 - Sep 12, 2004 - fixed bug in regards to AnimList during Apply Changes (forgot to use 'next unless' in pattern match)
#                         fixed bug in open_dialog_from_arg regarding menu deletion
#                         added Ctrl-J binding
#                         added stuntlist editing, improved animationlist editing
# 0.13 - Sep 13, 2004 - added auxilary File menu functions (TLK, GFFEditor, DLGEdit)
#                     - added extra field to hold DLGEditor info in the save_game function
# 1.0.0 - Sep 13, 2004 - v0.13 upgraded to 1.0.0 and released.
# 1.0.1 - Sep 13, 2004 - Fixed bug: if StuntList not present, causes corruption of DelayEntry field.
# 1.0.2 - Sep 17, 2004 - Added check in populate_children in case of malformed EntriesList or RepliesList Index field
#                      - Modified populate_tree function to check for errors in StartingList
#                      - commented out dialog boxes that said "GFF Editor not found" or "DLGEdit not found"
#                      - added 'return unless $gff' after every populate_children call in order to bubble back up
# 1.0.3 - Sep 18, 2004 - fixed bug: Comments not being stored
#                      - separated out functions: delete_[entries|replies|starting]list_structs_by_index
#                      - fixed bug in get_parent_child_link_structs that affected StartingList entries after one was deleted
#                      - updated delete_node_completely and cut_node functions to look at GFF object rather than tree for information
#                      - added Paste options to context menu when root node is selected
# 1.0.4 - Sep 20, 2004 - now requires GFF.pm v0.63 (psuedo-support for unicode in CEXOSTRING, RESREF, CEXOLOCSTRING)
# 1.0.5 - Oct 13, 2004 - in show_detail_widgets, added more checks to get_field_by_ix and added createField calls if field does not exist
#                        This is done to make DLGEditor more compatible with those DLG files created with KotOR Tool.
# to do
#       -- Test Drive Mode in GUI



my $compilation_date="Oct 13, 2004";
my $version='1.0.5';

use strict;
use Bioware::GFF 0.63;
use Bioware::TLK;
use Tk;
use Tk::DragDrop;
use Tk::DropSite;
use Tk::Tree;
use Tk::ItemStyle;
use Tk::LabEntry;
use Tk::DialogBox;

use Win32::FileOp;
use Win32::TieRegistry;
use bytes;
#
# begin subclassing of Tk::Tree widget
#
package Tk::MyTree;
use base qw(Tk::Tree);
Construct Tk::Widget "MyTree";
my $bail_out_of_ButtonRelease1;
sub anchorSet {}                      # disbaling the 'anchor'
sub Button1Motion {}                  # disabling the Drag event
sub ButtonRelease1 {
 my ($w, $Ev) = @_;
 if ($bail_out_of_ButtonRelease1) {   # this bit is custom to prevent
    $bail_out_of_ButtonRelease1=0;    # this event from occuring after a dbl-click
    return;
 }
 Tk::HList::ButtonRelease1($w, $Ev);
}

sub Double1
{
 my $w = shift;
 $bail_out_of_ButtonRelease1=1;       # set this flag to indicate this is a dbl-click
 Tk::HList::Double1($w);
}


package main;


sub mnuFile;
sub mnuEdit;
sub mnuHelp;
sub bindDump;
my %animation_num2desc=(10006=>'DEAD',             10028=>'TAUNT',         10029=>'GREETING',
                        10030=>'LISTEN',           10033=>'WORSHIP',       10034=>'SALUTE',
                        10035=>'BOW',              10038=>'TALK NORMAL',   10039=>'TALK PLEADING',
                        10040=>'TALK FORCEFUL',    10041=>'TALK LAUGHING', 10042=>'TALK SAD',
                        10044=>'VICTORY',          10055=>'SCRATCH HEAD',  10058=>'DRUNK',
                        10070=>'INJECT',           10120=>'FLIRT',         10121=>'USE COMPUTER (loop)',
                        10124=>'HORROR',           10125=>'USE COMPUTER',  10126=>'PERSUADE',
                        10127=>'ACTIVATE',         10137=>'SLEEP',         10139=>'PRONE',
                        10148=>'READY',            10149=>'PAUSE',         10150=>'CHOKE',
                        10154=>'TALK INJURED',     10155=>'LISTEN INJURED',10163=>'KNEEL TALK ANGRY',
                        10164=>'KNEEL TALK SAD',   10219=>'PUSHED BACK',   10220=>'FORCE JUMP',
                        10221=>'COLLAPSE CHOKING', 10222=>'COLLAPSE KNOCKOUT', 10223=>'FALL SLOWLY (RIGHT)',
                        10224=>'FALL SLOWLY (CENTER)', 10225=>'UNKNOWN #225',10226=>'UNKNOWN #226'
                       );
my %animation_desc2num=reverse %animation_num2desc;
my $path_to_dialog_dot_tlk;
my $current_dialog;
my $gff;
my $gff_dirty_flag=0;
my %entry_ixs;
my %reply_ixs;
my %tree_memory;
my @spawned_widgets;


my $view_conditionals;
my $view_scripts;
my $view_indices;
my $view_orphans;

my $dbl_clicked;

my %clipboard;

my $old_selection;

eval { my  $kotor_key= new Win32::TieRegistry "LMachine/Software/Bioware/SW/Kotor",             #read registry
          {Access=>Win32::TieRegistry::KEY_READ, Delimiter=>"/"};
        $path_to_dialog_dot_tlk= $kotor_key->GetValue("Path") };
unless (-e $path_to_dialog_dot_tlk."\\dialog.tlk") {
    exit unless ($path_to_dialog_dot_tlk=OpenDialog(
                                                    title=>'Locate Dialog.tlk file',
                                                    filters=>['Dialog.tlk file'=>'dialog.tlk'],
                                                    options=>OFN_FILEMUSTEXIST|OFN_HIDEREADONLY)
                 );
    $path_to_dialog_dot_tlk=~/(.*)\\/;
    $path_to_dialog_dot_tlk=$1;

}



#========================
# Begin Main Tk Elements
#========================
#  ----------------
#   Main Window
#  ----------------
my $mw=MainWindow->new(-title=>'DLGEditor');
eval {
   my $datafile="1.bmp";
   my $filename=PerlApp::extract_bound_file($datafile);
   my $image = $mw->Photo(-file => $filename, -format => 'bmp');
   $mw->Icon(-image => $image);
};

#  ----------------
#   MenuBar
#  ----------------
sub mnuFile;
sub mnuEdit;
sub mnuHelp;
sub mnuView;
sub create_new_dialog;
sub open_dialog;
sub close_dialog;
sub save_dialog;
sub save_as_dialog;
sub change_dialog_dot_tlk;
sub launch_gffeditor;
sub launch_dlgedit;
sub add_entry;
sub add_reply;
sub cut_node;
sub copy_node;
sub paste_node;
sub splash;

sub clear_tree;
sub populate_tree;
sub populate_children;
sub get_reply_text_by_index ($);
sub get_entry_text_by_index ($);
sub show_detail_widgets;
sub show_general_detail_widgets;

sub tree_click;
sub tree_rclick;
sub tree_dblclick;
sub stuntlist_rclick;
sub animation_list_rclick;

$mw->configure(-menu=>my $menubar=$mw->Menu(-tearoff=>0));
$menubar->cascade(-label=>'~File',-menuitems=>mnuFile,-tearoff=>0);
#$menubar->cascade(-label=>'~Edit',-menuitems=>mnuEdit,-tearoff=>0);
#$menubar->cascade(-label=>'~View',-menuitems=>mnuView,-tearoff=>0);
$menubar->cascade(-label=>'~Help',-menuitems=>mnuHelp,-tearoff=>0);


sub mnuFile {
    [
     ['command','~New',    -accelerator=>'Ctrl+N', -command=>sub { create_new_dialog() }],
     ['command','~Open...',   -accelerator=>'Ctrl+O', -command=>sub { open_dialog() }],
     ['command','~Revert',    -accelerator=>'Ctrl+Backspace', -command=>sub { open_dialog_from_arg($current_dialog) },-state=>'disabled'],
     ['command','~Close',     -accelerator=>'Ctrl+F4',-command=>sub { close_dialog()}, -state=>'disabled'],
     ['command','~Save',      -accelerator=>'Ctrl+S', -command=>sub { save_dialog() }, -state=>'disabled'],
     ['command','Save ~As...',-accelerator=>'Ctrl+A', -command=>sub { save_as_dialog () }, -state=>'disabled'],
     '',
     ['command','Change ~TLK File',-accelerator=>'Ctrl+T', -command=>sub { change_dialog_dot_tlk() }],
     ['command','Launch in ~GFF Editor',-accelerator=>'Ctrl+G', -command=>sub { launch_gffeditor() }],
     ['command','Launch in ~DLGEdit',-accelerator=>'Ctrl+D', -command=>sub { launch_dlgedit() }],
     ['command','E~xit',      -accelerator=>'Ctrl+Q', -command=>sub { close_dialog(); exit; }]
    ]

}
sub mnuEdit {
    [
     ['command','Add New ~Entry',-accelerator=>'Ctrl+E', -command=>sub { add_entry(undef) }, -state=>'disabled' ],
     ['command','Add New ~Reply',-accelerator=>'Ctrl+R', -command=>sub { add_reply(undef) } , -state=>'disabled'],
     '',
     ['command','~Break Link With Parent',-accelerator=>'Ctrl+End',-command=>sub { break_link_with_parent(); refresh_tree(); } , -state=>'disabled'],
     ['command','~Delete Node Completely',-accelerator=>'Ctrl+Delete',-command=>sub { delete_node() } , -state=>'disabled'],
     '',
     ['command','Cu~t Node', -accelerator=>'Ctrl+X',-command=>sub { cut_node() }, -state=>'disabled'],
     ['command','~Copy Node', -accelerator=>'Ctrl+C',-command=>sub { copy_node() }, -state=>'disabled'],
     ['command','~Paste Node As Copy', -accelerator=>'Ctrl+V',-command=>sub { paste_node('copy') }, -state=>'disabled'],
     ['command','Paste Node As ~New', -accelerator=>'Ctrl+Insert',-command=>sub { paste_node('new') }, -state=>'disabled']
    ]
}
sub mnuView {
    [
     ['checkbutton','Show ~Conditionals in Tree',-accelerator=>'F2',-variable=>\$view_conditionals,-command=>sub { refresh_tree_gentle() }, ],
     ['checkbutton','Show ~Scripts in Tree',-accelerator=>'F3',-variable=>\$view_scripts,-command=>sub { refresh_tree_gentle() }, ],
     ['checkbutton','Show ~Indices in Tree',-accelerator=>'F4',-variable=>\$view_indices,-command=>sub { refresh_tree_gentle() }, ],
     ['checkbutton','Show ~Orphaned Replies/Entries',-accelerator=>'F5',-variable=>\$view_orphans,-command=> sub { refresh_tree() }],
     '',
     ['command','~Expand All', -accelerator=>']',-command=>sub { expand_tree();}],
     ['command','~Fold All', -accelerator=>'[',-command=>sub { fold_tree();}],
    ]
}
sub mnuHelp {
    [
     ['command','~About DLGEditor',-command=>sub { splash() } ]
    ]
}


#  -------------------
#   Upper Frame (Tree)
#  -------------------
my $upper_frame = $mw->Frame(-borderwidth=>2,-relief=>'groove')->pack(-expand=>1,-fill=>'both',-side=>'top');

#  -------------------
#   Lower Frame (Info)
#  -------------------
my $lower_frame = $mw->Frame(-borderwidth=>2,-relief=>'groove')->pack(-expand=>1,-fill=>'both',-side=>'bottom');
#  -------------------
#    Tree
#  -------------------
my $tree_bgcolor='#E7E7E7';
my $tree=$upper_frame->Scrolled("MyTree",
                                -scrollbars=>'se',
                                -separator=>'#',
                                -background=>$tree_bgcolor,
                                -selectbackground=>'#FFFFF0',
                                -selectmode=> 'single',
                                -browsecmd=>sub {my $treeitem=shift;
                                                 tree_click($treeitem);
                                                 },
                                -command=>sub { my $treeitem=shift;
                                               tree_dblclick($treeitem);
                                               }
                                )->pack(-fill=>'both',-expand=>1);

#bind right click
$tree->bind('<ButtonPress-3>'=>sub { my $treeitem=shift; tree_rclick($treeitem); } );

# -------------------------
#  Bind Keystrokes to Menu
# -------------------------
#$mw->bind('<KeyPress>'=>\&print_keysym); #for binding debugging only
$mw->bind('<Control-n>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('New',-state);
   if ($state eq 'normal') { create_new_dialog(); }
 });
$mw->bind('<Control-o>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Open...',-state);
   if ($state eq 'normal') { open_dialog(); }
 });
$mw->bind('<Control-F4>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Close',-state);
   if ($state eq 'normal') { close_dialog(); }
 });

$mw->bind('<Control-BackSpace>'=>sub {
   my $file_menu=$menubar->entrycget('File',-menu);
   my $state=$file_menu->entrycget('Revert',-state);
   if ($state eq 'normal') { open_dialog_from_arg($current_dialog) }
 });
$mw->bind('<Control-s>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Save',-state);
   if ($state eq 'normal') { save_dialog(); }
 });
$mw->bind('<Control-a>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Save As...',-state);
   if ($state eq 'normal') { save_as_dialog(); }
 });
$mw->bind('<Control-t>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Change TLK File',-state);
   if ($state eq 'normal') { change_dialog_dot_tlk(); }
 });
$mw->bind('<Control-g>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Launch in GFF Editor',-state);
   if ($state eq 'normal') { launch_gffeditor(); }
 });
$mw->bind('<Control-d>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Launch in DLGEdit',-state);
   if ($state eq 'normal') { launch_dlgedit(); }
 });
$mw->bind('<Control-q>'=>sub {
   my $edit_menu=$menubar->entrycget('File',-menu);
   my $state=$edit_menu->entrycget('Exit',-state);
   if ($state eq 'normal') { exit; }
 });
$mw->bind('<Control-e>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Add New Entry',-state);
   if ($state eq 'normal') { add_entry(undef) }
 });
$mw->bind('<Control-r>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Add New Reply',-state);
   if ($state eq 'normal') { add_reply(undef) }
 });
$mw->bind('<Control-Delete>'=>sub {
   $mw->bind('<KeyRelease-Delete>'=>sub{});  # turn off the competing binding
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Delete Node Completely',-state);
   if ($state eq 'normal') { delete_node(); }
 });
$mw->bind('<Control-End>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Break Link With Parent',-state);
   if ($state eq 'normal') { break_link_with_parent(); refresh_tree();  }
 });

$tree->bind('<Control-x>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Cut Node',-state);
   if ($state eq 'normal') { cut_node() }
 });

$tree->bind('<Control-v>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Paste Node As Copy',-state);
   if ($state eq 'normal') { paste_node('copy') }
   else { my $other_state=$edit_menu->entrycget('Paste Node As New',-state);
      if ($other_state eq 'normal') {
         my $d = $mw->DialogBox(-title => "Paste As New?",-buttons => ["OK", "Cancel"]);
         $d->add('Label', -text=>"Cannot Paste as Copy.\n  Paste as New (Ctrl+Insert) instead?")->pack;
         my $answer=$d->Show(-popover  => 'cursor', -popanchor => 'nw');
         if ($answer eq 'OK') { paste_node('new') }
      }
   }
 });
$tree->bind('<Control-c>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Copy Node',-state);
   if ($state eq 'normal') { copy_node() }
 });
$tree->bind('<Control-Insert>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $edit_menu=$menubar->entrycget('Edit',-menu);
   my $state=$edit_menu->entrycget('Paste Node As New',-state);
   if ($state eq 'normal') { paste_node('new') }
});
$tree->bind('<Key-F2>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $view_menu=$menubar->entrycget('View',-menu);
   my $state=$view_menu->entrycget('Show Conditionals in Tree',-state);
   if ($state eq 'normal') { $view_conditionals = !$view_conditionals;  refresh_tree_gentle(); }
 });
$tree->bind('<Key-F3>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $view_menu=$menubar->entrycget('View',-menu);
   my $state=$view_menu->entrycget('Show Scripts in Tree',-state);
   if ($state eq 'normal') { $view_scripts =!$view_scripts; refresh_tree_gentle(); }
 });
$tree->bind('<Key-F4>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $view_menu=$menubar->entrycget('View',-menu);
   my $state=$view_menu->entrycget('Show Indices in Tree',-state);
   if ($state eq 'normal') { $view_indices =!$view_indices;refresh_tree_gentle(); }
 });
$tree->bind('<Key-F5>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $view_menu=$menubar->entrycget('View',-menu);
   my $state=$view_menu->entrycget('Show Orphaned Replies/Entries',-state);
   if ($state eq 'normal') { $view_orphans= !$view_orphans; refresh_tree(); }
 });
$tree->bind('<Key-[>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $view_menu=$menubar->entrycget('View',-menu);
   my $state=$view_menu->entrycget('Fold All',-state);
   if ($state eq 'normal') { fold_tree(); }
 });
$tree->bind('<Key-]>'=>sub {
   return unless $menubar->entrycget('1',-label) =~ /Edit/;
   my $view_menu=$menubar->entrycget('View',-menu);
   my $state=$view_menu->entrycget('Expand All',-state);
   if ($state eq 'normal') { expand_tree(); }
 });
$tree->bind('<Control-j>'=>sub {
   tree_dblclick(($tree->infoSelection)[0]);
});

#$tree->add("#",-text=>'(empty)');

# ---------------
#  Styles for Tree

# note for some reason the selectbackground doesn't seem to be working...
my $entrystyle = $tree->ItemStyle('text',-background=>$tree_bgcolor,-foreground=>'#FF0000',-selectforeground=>'#9F0000',-selectbackground=>'#FFD0D0',-font=>['Century','8','bold']);
my $replystyle = $tree->ItemStyle('text',-background=>$tree_bgcolor,-foreground=>'#0000FF',-selectforeground=>'#00009F',-selectbackground=>'#D0D0FF',-font=>['Century','8','bold']);
my $rootstyle=   $tree->ItemStyle('text',-background=>$tree_bgcolor,-foreground=>'#000000',-selectforeground=>'#000000',-selectbackground=>'#D0D0D0',-font=>['Century','8','bold']);
my $orphan_entrystyle=$tree->ItemStyle('text',-background=>$tree_bgcolor,-foreground=>'#800080',-selectforeground=>'#800080',-selectbackground=>'#FFD0D0',-font=>['Century','8','bold']);
my $orphan_replystyle=$tree->ItemStyle('text',-background=>$tree_bgcolor,-foreground=>'#800080',-selectforeground=>'#800080',-selectbackground=>'#FFD0D0',-font=>['Century','8','bold']);
$mw->geometry('900x600');

if (scalar @ARGV > 0) {
    #print "trying to open $ARGV[0]";
    #user is trying to open a dialog file by passing it as an argument
    open_dialog_from_arg($ARGV[0]);
}

MainLoop;


#=======================
# Menu subroutines
#=======================


sub create_new_dialog{
    last_chance_to_save();
    clear_tree();
    destroy_old_widgets();

    $gff=Bioware::GFF->new('sig'=>'DLG ', 'version'=>'V3.2');
    $gff->{Main}->createField('Type'=>FIELD_DWORD,'Label'=>'DelayEntry','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_DWORD,'Label'=>'DelayReply','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_DWORD,'Label'=>'NumWords','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_RESREF,'Label'=>'EndConverAbort');
    $gff->{Main}->createField('Type'=>FIELD_RESREF,'Label'=>'EndConversation');
    $gff->{Main}->createField('Type'=>FIELD_BYTE,'Label'=>'Skippable','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_RESREF,'Label'=>'AmbientTrack');
    $gff->{Main}->createField('Type'=>FIELD_BYTE,'Label'=>'AnimatedCut','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_RESREF,'Label'=>'CameraModel');
    $gff->{Main}->createField('Type'=>FIELD_BYTE,'Label'=>'ComputerType','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_INT,'Label'=>'ConversationType','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_LIST,'Label'=>'EntryList','Value'=>[]);
    $gff->{Main}->createField('Type'=>FIELD_BYTE,'Label'=>'OldHitCheck','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_LIST,'Label'=>'ReplyList','Value'=>[]);
    $gff->{Main}->createField('Type'=>FIELD_LIST,'Label'=>'StartingList','Value'=>[]);
    $gff->{Main}->createField('Type'=>FIELD_BYTE,'Label'=>'UnequipHItem','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_BYTE,'Label'=>'UnequipItems','Value'=>0);
    $gff->{Main}->createField('Type'=>FIELD_LIST,'Label'=>'StuntList','Value'=>[]);
    $gff->{Main}->createField('Type'=>FIELD_CEXOSTRING,'Label'=>'VO_ID');
    $current_dialog='';
    populate_tree();
    enable_menus();

}
sub open_dialog{
    last_chance_to_save();
    return unless ($current_dialog=OpenDialog(title=>'Select a .dlg file to open',
                                              filters=>['Dialog files'=>'*.dlg'],
                                              options=>OFN_FILEMUSTEXIST|OFN_HIDEREADONLY));
    clear_tree();
    destroy_old_widgets();
    $gff=Bioware::GFF->new();
    return unless ($gff->read_gff_file($current_dialog));
    return unless ($gff->{'sig'} eq 'DLG ');
    populate_tree();
    enable_menus();
}
sub open_dialog_from_arg {
    $current_dialog=shift;
    disable_menus();
    $gff_dirty_flag=0;
    clear_tree();
    destroy_old_widgets();
    $gff=Bioware::GFF->new();
    return unless ($gff->read_gff_file($current_dialog));
    return unless ($gff->{'sig'} eq 'DLG ');
    populate_tree();
    enable_menus();
}
sub expand_tree{ #this will open all nodes
    my $treeitem='#';
    while ($treeitem=$tree->infoNext($treeitem)) {
        $tree->open($treeitem);
    }
    $tree->open('#');
}
sub fold_tree{ #this will close all nodes
    my $treeitem='#';
    while ($treeitem=$tree->infoNext($treeitem)) {
        $tree->close($treeitem);
    }
    #$tree->close('#');
}
sub close_dialog{ #closing the .dlg file, (really just emptying memory)
    last_chance_to_save();
    clear_tree();
    destroy_old_widgets();
    $gff=undef;
    $gff_dirty_flag=0;
    disable_menus();
}
sub save_dialog{
   $gff_dirty_flag=0;
   unless ($current_dialog) { save_as_dialog(); }
   if ($current_dialog) {
      my ($sec,$min,$hour,$mday,$mon,$year,undef,undef,undef)=localtime;
      my @months=qw/Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec/;
      my $info=sprintf('v%s %s LastEdit: %2.2u-%s-%2.2u %2.2u:%2.2u:%2.2u',$version,$compilation_date,$mday,$months[$mon],$year-100,$hour,$min,$sec);
      my $info_ix=$gff->{Main}->get_field_ix_by_label('EditorInfo');
      if (defined $info_ix) {
       $gff->{Main}{Fields}[$info_ix]{Value}=$info;
      }
      else {
        $gff->{Main}->createField('Type'=>FIELD_CEXOSTRING,'Label'=>'EditorInfo','Value'=>$info);
      }
      $gff->write_gff_file($current_dialog);
   }
}
sub save_as_dialog{ #save as dialog box
    my %parms=(
                title => "Save Output File As", handle=>0,
                filters => { 'Dialog Files' => '*.dlg', 'All Files' => '*.*'},
                filename=>$current_dialog,
                options =>  OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT);
    $current_dialog = SaveAsDialog \%parms;
    if ($current_dialog) {
        save_dialog();
        $tree->entryconfigure('#',-text=>$current_dialog);
    }
}
sub add_entry{
    my $treeitem=shift;                                      # in case add_reply or paste_node called this function, use its item
    unless ($treeitem) {($treeitem)=$tree->infoSelection();} # if not, then get current tree selection
    unless ($treeitem) { $treeitem='#' }                     # if nothing selected, choose root
    my $lastbranch=(split /#/,$treeitem)[-1];                # the lastbranch will be parent
    if ($lastbranch =~ /E/) {                                # user wants to add an entry underneath an entry...
       $treeitem=add_reply($treeitem);                       # we have to add a reply first (returns new treeitem)
       $lastbranch=(split /#/,$treeitem)[-1];                # get new lastbranch
    }
    $lastbranch=~/(\d+)/;
    my $parent_ix=$1;
    my $entry_ix=$gff->{Main}->get_field_ix_by_label('EntryList');
    my @structlist=@{$gff->{Main}{Fields}[$entry_ix]{Value}};
    my $new_struct_ix=scalar @structlist;
    my $new_struct=Bioware::GFF::Struct->new('ID'=>$new_struct_ix);
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Speaker'        ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_LIST,         'Label'=>'AnimList'       ,'Value'=>[]);
    $new_struct->createField('Type'=>FIELD_CEXOLOCSTRING,'Label'=>'Text'           ,'StringRef'=>-1,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_RESREF,       'Label'=>'VO_ResRef'      ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_RESREF,       'Label'=>'Script'         ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_DWORD,        'Label'=>'Delay'          ,'Value'=>-1);
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Comment'        ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_RESREF,       'Label'=>'Sound'          ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Quest'          ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_INT,          'Label'=>'PlotIndex'      ,'Value'=>-1);
    $new_struct->createField('Type'=>FIELD_FLOAT,        'Label'=>'PlotXPPercentage','Value'=>1);
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Listener'       ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_DWORD,        'Label'=>'WaitFlags'      ,'Value'=>0);
    $new_struct->createField('Type'=>FIELD_DWORD,        'Label'=>'CameraAngle'    ,'Value'=>0);
    $new_struct->createField('Type'=>FIELD_BYTE,         'Label'=>'FadeType'       ,'Value'=>0);
    $new_struct->createField('Type'=>FIELD_LIST,         'Label'=>'RepliesList'    ,'Value'=>[]);
    $new_struct->createField('Type'=>FIELD_BYTE,         'Label'=>'SoundExists'    ,'Value'=>0);
    push @{$gff->{Main}{Fields}[$entry_ix]{Value}},$new_struct;    # new entry has been added to gff

                                                                   # we now need to link the parent->child
    my $entrieslist_arr_ref;
    if ($treeitem eq '#') {                                        # special case for StartingList
        $entrieslist_arr_ref = $gff->{Main}{Fields}[
                                           $gff->{Main}->get_field_ix_by_label('StartingList')]{Value};
    }
    else {
        my $parent_struct=$gff->{Main}{Fields}[
                                           $gff->{Main}->get_field_ix_by_label('ReplyList')
                                           ]{Value}[$parent_ix];
        $entrieslist_arr_ref=$parent_struct->{Fields}[              # get entrieslist of parent
                                                     $parent_struct->get_field_ix_by_label('EntriesList')
                                                    ]{Value};
    }
    my $new_pstruct_id = scalar @$entrieslist_arr_ref;
    my $new_pstruct=Bioware::GFF::Struct->new('ID'=>$new_pstruct_id); # create new entrieslist struct
    $new_pstruct->createField('Type'=>FIELD_DWORD,'Label'=>'Index','Value'=>$new_struct_ix);
    $new_pstruct->createField('Type'=>FIELD_RESREF,'Label'=>'Active','Value'=>'');
    unless ($treeitem eq '#') {
        $new_pstruct->createField('Type'=>FIELD_BYTE,'Label'=>'IsChild','Value'=>0);
    }
    push @$entrieslist_arr_ref,$new_pstruct;                       # add entrieslist struct to parent

    refresh_tree();                                               # show our changes
    my $new_treeitem=$treeitem;
    if ($treeitem=~/#$/) {
        $new_treeitem.='E'.$new_struct_ix;
    }
    else {
        $new_treeitem.='#E'.$new_struct_ix;
    }
    $tree->selectionSet($treeitem);
    $tree->see($new_treeitem);
    gff_updated();
    return $new_treeitem;                                          # return our new treeitem
}
sub add_reply{
    my $treeitem=shift;                                      # in case add_entry called this function, use its item
    unless ($treeitem) {($treeitem)=$tree->infoSelection();} # if not, then get current tree selection
    unless ($treeitem) { $treeitem='#' }                     # if nothing selected, choose root
    my $lastbranch=(split /#/,$treeitem)[-1];
    if (($lastbranch =~ /R/) || ($treeitem eq '#')) {        # user wants to add an reply underneath a reply or startinglist...
       $treeitem=add_entry($treeitem);                       # we have to add an entry first (returns new treeitem)
       $lastbranch=(split /#/,$treeitem)[-1];                # get new lastbranch
    }
    $lastbranch=~/(\d+)/;
    my $parent_ix=$1;

    my $reply_ix=$gff->{Main}->get_field_ix_by_label('ReplyList');
    my @structlist=@{$gff->{Main}{Fields}[$reply_ix]{Value}};
    my $new_struct_ix=scalar @structlist;
    my $new_struct=Bioware::GFF::Struct->new('ID'=>$new_struct_ix);
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Listener'       ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_LIST,         'Label'=>'AnimList'       ,'Value'=>[]);
    $new_struct->createField('Type'=>FIELD_CEXOLOCSTRING,'Label'=>'Text'           ,'StringRef'=>-1,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_RESREF,       'Label'=>'VO_ResRef'      ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_RESREF,       'Label'=>'Script'         ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_DWORD,        'Label'=>'Delay'          ,'Value'=>-1);
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Comment'        ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_RESREF,       'Label'=>'Sound'          ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Quest'          ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_INT,          'Label'=>'PlotIndex'      ,'Value'=>-1);
    $new_struct->createField('Type'=>FIELD_FLOAT,        'Label'=>'PlotXPPercentage','Value'=>1);
    $new_struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Listener'       ,'Value'=>'');
    $new_struct->createField('Type'=>FIELD_DWORD,        'Label'=>'WaitFlags'      ,'Value'=>0);
    $new_struct->createField('Type'=>FIELD_DWORD,        'Label'=>'CameraAngle'    ,'Value'=>0);
    $new_struct->createField('Type'=>FIELD_BYTE,         'Label'=>'FadeType'       ,'Value'=>0);
    $new_struct->createField('Type'=>FIELD_LIST,         'Label'=>'EntriesList'    ,'Value'=>[]);
    $new_struct->createField('Type'=>FIELD_BYTE,         'Label'=>'SoundExists'    ,'Value'=>0);
    push @{$gff->{Main}{Fields}[$reply_ix]{Value}},$new_struct;    # new reply has been added to gff

    my $parent_struct=$gff->{Main}{Fields}[                        # we now need to link the parent->child
                                           $gff->{Main}->get_field_ix_by_label('EntryList')
                                           ]{Value}[$parent_ix];
    my $replieslist_arr_ref=$parent_struct->{Fields}[              # get Replieslist of parent
                                                     $parent_struct->get_field_ix_by_label('RepliesList')
                                                    ]{Value};
    my $new_pstruct_id = scalar @$replieslist_arr_ref;
    my $new_pstruct=Bioware::GFF::Struct->new('ID'=>$new_pstruct_id); # create new replieslist struct
    $new_pstruct->createField('Type'=>FIELD_DWORD,'Label'=>'Index','Value'=>$new_struct_ix);
    $new_pstruct->createField('Type'=>FIELD_RESREF,'Label'=>'Active','Value'=>'');
    $new_pstruct->createField('Type'=>FIELD_BYTE,'Label'=>'IsChild','Value'=>0);
    push @$replieslist_arr_ref,$new_pstruct;                       # add replieslist struct to parent

    refresh_tree();                                                # show our changes
    my $new_treeitem=$treeitem;
    if ($treeitem=~/#$/) {
        $new_treeitem.='R'.$new_struct_ix;
    }
    else    {
    $new_treeitem.='#R'.$new_struct_ix;
    }
    $tree->selectionSet($treeitem);
    $tree->see($new_treeitem);
    gff_updated();
    return $new_treeitem;                                           # return our new treeitem
}
sub break_link_with_parent {
    my $treeitem=shift;
    unless ($treeitem) { $treeitem=$tree->infoSelection }
    return unless ($treeitem);                                      # return if nothing selected
    return if ($treeitem eq '#');                                   # return if root selected
    my $lastbranch=(split /#/, $treeitem)[-1];
    return if ($lastbranch =~ /O/);                                 # return if orphan
    my @newarr;
    my ($parent_type,$parent_struct,$parent_struct_index, $parent_substruct, $parent_substruct_index,$parent_substruct_arr_ref)=get_parent_child_link_structs($treeitem);
                                                                    # we will reconstruct the link array
    for (my $i=0; $i<scalar @$parent_substruct_arr_ref; $i++) {     # loop through original array
        my $struct = $$parent_substruct_arr_ref[$i];                # get each struct
        $struct->{'ID'}=scalar @newarr;                             # keep struct id in sync with new array
        if ($parent_substruct_index != $i) {                        # if this struct is not the one to omit
            push @newarr, $struct;                                  # then add it to our new array
        }
    }
    if ($parent_type eq 'S') {
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StartingList')]{Value}=[@newarr];
    }
    elsif ($parent_type eq 'E') {
        $parent_struct->{Fields}[$parent_struct->get_field_ix_by_label('RepliesList')]{Value}=[@newarr];
    }
    elsif ($parent_type eq 'R') {
        $parent_struct->{Fields}[$parent_struct->get_field_ix_by_label('EntriesList')]{Value}=[@newarr];
    }
    gff_updated();
    refresh_tree();
    destroy_old_widgets();
}
sub delete_node {
   my $treeitem=shift;
   unless ($treeitem) { ($treeitem)=$tree->infoSelection }
   return unless ($treeitem);                                      # return if nothing selected
   return if ($treeitem eq '#');                                   # return if root selected
   my $lastbranch=(split /#/, $treeitem)[-1];
   $lastbranch=~/(\d+)/;
   my $index=$1;

   # break link with all parents...
   if ($lastbranch =~ /E/) {
      delete_startinglist_structs_by_index($index);
      delete_entrieslist_structs_by_index($index);
   }
   elsif ($lastbranch =~/R/) {
      delete_replieslist_structs_by_index($index);
   }

    # delete node from gff...
    my $struct_arr_ref;
    if ($lastbranch =~/E/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
    }
    elsif ($lastbranch =~/R/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
    }
    my @newarr=();
    for (my $i=0; $i<scalar @$struct_arr_ref; $i++) {
        $$struct_arr_ref[$i]->{'ID'}=scalar @newarr;
        unless ($i==$index){
            push @newarr,$$struct_arr_ref[$i];
        }
    }
    if ($lastbranch =~/E/) {
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value}=[@newarr];
    }
    elsif ($lastbranch =~/R/) {
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value}=[@newarr];
    }

    # update those parents with links to children that got shifted...
    my $other_parents_arr_ref;
    if ($lastbranch=~/E/) {                                        # parents of an entry are replies
        $other_parents_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
    }
    elsif ($lastbranch=~/R/) {                                     # parents of a reply are entries
        $other_parents_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
    }
    for my $other_parent (@$other_parents_arr_ref) {               # look at this parent
        my $other_parent_child_link_arr_ref;
        if ($lastbranch=~/E/) {                                    # links of parents of entries are entrieslists
            $other_parent_child_link_arr_ref=$other_parent->{Fields}[$other_parent->get_field_ix_by_label('EntriesList')]{Value};
        }
        elsif ($lastbranch=~/R/) {                                 # links of parents of replies are replieslists
            $other_parent_child_link_arr_ref=$other_parent->{Fields}[$other_parent->get_field_ix_by_label('RepliesList')]{Value};
        }
        for my $other_parent_child_link (@$other_parent_child_link_arr_ref) { # look at the links of this parent
            my $link_ix=$other_parent_child_link->{Fields}[$other_parent_child_link->get_field_ix_by_label('Index')]{Value};
            if ($link_ix>$index) {                                # update any link to a child that got shifted up because of our cut
                $other_parent_child_link->{Fields}[$other_parent_child_link->get_field_ix_by_label('Index')]{Value}=$link_ix-1;
            }
        }
    }
    # and we can't forget about updating the StartingList if this was an Entry
    if ($lastbranch =~ /E/) {
        my $startinglist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StartingList')]{Value};
        for my $sl_struct (@$startinglist_arr_ref) {
            my $link_ix=$sl_struct->{Fields}[$sl_struct->get_field_ix_by_label('Index')]{Value};
            if ($link_ix>$index) {
                $sl_struct->{Fields}[$sl_struct->get_field_ix_by_label('Index')]{Value}=$link_ix-1;
            }
        }
    }
    gff_updated();
    refresh_tree();
    destroy_old_widgets();
}
sub cut_node{
   return unless (my ($treeitem)=$tree->infoSelection);            # return if nothing selected
   return if ($treeitem eq '#');                                   # return if root selected
   my $lastbranch=(split /#/, $treeitem)[-1];
   $lastbranch=~/(\d+)/;
   my $index=$1;

   #first break parent-child link
   break_link_with_parent($treeitem);

   #check to see if there are other parents to this entry/reply
   #if so, then just save the entry/reply index into memory (symbolic)
   #otherwise, we need to remove structure and shuffle other links
   my $occurances;

   if ($lastbranch =~ /E/) {
      my $startinglist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StartingList')]{Value};
      for my $startinglist_struct (@$startinglist_arr_ref) {
         if ($index == $startinglist_struct->{Fields}[$startinglist_struct->get_field_ix_by_label('Index')]{Value}) {
            $occurances++;
            last;
         }
      }
      unless ($occurances) {
         my $replylist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
         for my $reply_struct (@$replylist_arr_ref) {
            my $entrieslist_arr_ref=$reply_struct->{Fields}[$reply_struct->get_field_ix_by_label('EntriesList')]{Value};
            for my $entrieslist_struct (@$entrieslist_arr_ref) {
               if ($index == $entrieslist_struct->{Fields}[$entrieslist_struct->get_field_ix_by_label('Index')]{Value}) {
                  $occurances++;
                  last;
               }
            }
            last if $occurances;
         }
      }
   }
   elsif ($lastbranch =~ /R/) {
      my $entrylist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
      for my $entry_struct (@$entrylist_arr_ref) {
         my $replieslist_arr_ref=$entry_struct->{Fields}[$entry_struct->get_field_ix_by_label('RepliesList')]{Value};
         for my $replieslist_struct (@$replieslist_arr_ref) {
            if ($index == $replieslist_struct->{Fields}[$replieslist_struct->get_field_ix_by_label('Index')]{Value}) {
               $occurances++;
               last;
            }
         }
         last if $occurances;
      }
   }

   if ($occurances) { # symbolic
      my $struct_arr_ref;
      if ($lastbranch =~/E/) {
          $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
      }
      elsif ($lastbranch =~/R/) {
          $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
      }
      %clipboard=();
      $clipboard{'symbol'}=1;
      $clipboard{'struct'}=$$struct_arr_ref[$index];
      $clipboard{'old treeitem'}=$treeitem;
      $clipboard{'old lastbranch'}=$lastbranch;
      my $edit_menu=$menubar->entrycget('Edit',-menu);
      $edit_menu->entryconfigure('Paste Node As Copy',-state=>'normal');
      $edit_menu->entryconfigure('Paste Node As New',-state=>'normal');
      gff_updated();
      refresh_tree();
      destroy_old_widgets();

      return;
   }

    #put structure represented by treeitem into memory
    my $struct_arr_ref;
    if ($lastbranch =~/E/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
    }
    elsif ($lastbranch =~/R/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
    }

    %clipboard=();
    $clipboard{'symbol'}=0;
    $clipboard{'struct'}=$$struct_arr_ref[$index];
    $clipboard{'old treeitem'}=$treeitem;
    $clipboard{'old lastbranch'}=$lastbranch;


    #remove structure from GFF object...
    my @newarr=();
    for (my $i=0; $i<scalar @$struct_arr_ref; $i++) {
        $$struct_arr_ref[$i]->{'ID'}=scalar @newarr;
        unless ($i==$index){
            push @newarr,$$struct_arr_ref[$i];
        }
    }
    if ($lastbranch =~/E/) {
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value}=[@newarr];
    }
    elsif ($lastbranch =~/R/) {
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value}=[@newarr];
    }

    #update all pointers that point to structures that came 'after' the cut structure
    my $other_parents_arr_ref;
    if ($lastbranch=~/E/) {                                        # parents of an entry are replies
        $other_parents_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
    }
    elsif ($lastbranch=~/R/) {                                     # parents of a reply are entries
        $other_parents_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
    }
    for my $other_parent (@$other_parents_arr_ref) {               # look at this parent
        my $other_parent_child_link_arr_ref;
        if ($lastbranch=~/E/) {                                    # links of parents of entries are entrieslists
            $other_parent_child_link_arr_ref=$other_parent->{Fields}[$other_parent->get_field_ix_by_label('EntriesList')]{Value};
        }
        elsif ($lastbranch=~/R/) {                                 # links of parents of replies are replieslists
            $other_parent_child_link_arr_ref=$other_parent->{Fields}[$other_parent->get_field_ix_by_label('RepliesList')]{Value};
        }
        for my $other_parent_child_link (@$other_parent_child_link_arr_ref) { # look at the links of this parent
            my $link_ix=$other_parent_child_link->{Fields}[$other_parent_child_link->get_field_ix_by_label('Index')]{Value};
            if ($link_ix>$index) {                                # update any link to a child that got shifted up because of our cut
                $other_parent_child_link->{Fields}[$other_parent_child_link->get_field_ix_by_label('Index')]{Value}=$link_ix-1;
            }
        }
    }
    # and we can't forget about updating the StartingList if this was an Entry
    if ($lastbranch =~ /E/) {
        my $startinglist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StartingList')]{Value};
        for my $sl_struct (@$startinglist_arr_ref) {
            my $link_ix=$sl_struct->{Fields}[$sl_struct->get_field_ix_by_label('Index')]{Value};
            if ($link_ix>$index) {
                $sl_struct->{Fields}[$sl_struct->get_field_ix_by_label('Index')]{Value}=$link_ix-1;
            }
        }
    }

    my $edit_menu=$menubar->entrycget('Edit',-menu);
    $edit_menu->entryconfigure('Paste Node As New',-state=>'normal');
    $edit_menu->entryconfigure('Paste Node As Copy',-state=>'disabled');
    gff_updated();
    refresh_tree();
    destroy_old_widgets();
}
sub copy_node{
    return unless (my ($treeitem)=$tree->infoSelection);            # return if nothing selected
    return if ($treeitem eq '#');                                   # return if root selected
    my $lastbranch=(split /#/, $treeitem)[-1];
    $lastbranch=~/(\d+)/;
    my $index=$1;

    #put structure represented by treeitem into memory
    my $struct_arr_ref;
    if ($lastbranch =~/E/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
    }
    elsif ($lastbranch =~/R/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
    }

    %clipboard=();
    $clipboard{'symbol'}=1;
    $clipboard{'old treeitem'}=$treeitem;
    $clipboard{'old lastbranch'}=$lastbranch;
    $clipboard{'struct'}=$$struct_arr_ref[$index];

    my $edit_menu=$menubar->entrycget('Edit',-menu);
    $edit_menu->entryconfigure('Paste Node As Copy',-state=>'normal');
    $edit_menu->entryconfigure('Paste Node As New',-state=>'normal');
}
sub paste_node{
    my $paste_mode=shift;
    return unless (my ($treeitem)=$tree->infoSelection);            # return if nothing selected
    my $lastbranch=(split /#/,$treeitem)[-1];
    $lastbranch=~/(\d+)/;
    my $index=$1;

    #if user wants to paste as copy, then we will add a parental link and be done
    #otherwise, we will call add_entry or add_reply, receive back the new treeitem,
    #and then substitute our struct from the clipboard into the new struct represented by the new treeitem

    my $struct_arr_ref;
    my $struct;
    my $substruct_arr_ref;

    if ($treeitem eq '#') {
        $substruct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StartingList')]{Value};
    }
    if ($lastbranch=~/E/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
        $struct=$$struct_arr_ref[$index];
        $substruct_arr_ref=$struct->{Fields}[$struct->get_field_ix_by_label('RepliesList')]{Value};
    }
    elsif ($lastbranch=~/R/) {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
        $struct=$$struct_arr_ref[$index];
        $substruct_arr_ref=$struct->{Fields}[$struct->get_field_ix_by_label('EntriesList')]{Value};
    }

    if ($paste_mode eq 'copy'){
        #add a new link struct to parent with index from our clipboard
        if ($treeitem eq '#') { #pasting to StartingList
            if ($clipboard{'old lastbranch'}=~/R/) {   # sigh, we have to create a dummy entry first...
                $treeitem = add_entry($treeitem);      # treeitem is updated to rep. the new entry
                $lastbranch=(split /#/,$treeitem)[-1]; # lastbranch is updated
                $lastbranch=~/(\d+)/;
                $index=$1;                             # index is updated
                $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
                $struct=$$struct_arr_ref[$index];
                $substruct_arr_ref=$struct->{Fields}[$struct->get_field_ix_by_label('RepliesList')]{Value};
            }
        }
        elsif ($lastbranch =~ /E/) {
            if ($clipboard{'old lastbranch'}=~/E/) {   # sigh, we have to create a dummy reply first...
                $treeitem = add_reply($treeitem);      # treeitem is updated to rep. the new reply
                $lastbranch=(split /#/,$treeitem)[-1]; # lastbranch is updated
                $lastbranch=~/(\d+)/;
                $index=$1;                             # index is updated
                $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
                $struct=$$struct_arr_ref[$index];
                $substruct_arr_ref=$struct->{Fields}[$struct->get_field_ix_by_label('EntriesList')]{Value};
            }
        }
        elsif ($lastbranch =~ /R/) {
            if ($clipboard{'old lastbranch'}=~/R/) {   # sigh, we have to create a dummy entry first...
                $treeitem = add_entry($treeitem);      # treeitem is updated to rep. the new entry
                $lastbranch=(split /#/,$treeitem)[-1]; # lastbranch is updated
                $lastbranch=~/(\d+)/;
                $index=$1;                             # index is updated
                $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
                $struct=$$struct_arr_ref[$index];
                $substruct_arr_ref=$struct->{Fields}[$struct->get_field_ix_by_label('RepliesList')]{Value};
            }
        }
        # create new link to child
        my $link_struct_id = scalar @$substruct_arr_ref;
        my $link_struct=Bioware::GFF::Struct->new('ID'=>$link_struct_id);
        $clipboard{'old lastbranch'}=~/(\d+)/;
        my $child_struct_ix =$1;
        $link_struct->createField('Type'=>FIELD_DWORD,'Label'=>'Index','Value'=>$child_struct_ix);
        $link_struct->createField('Type'=>FIELD_RESREF,'Label'=>'Active','Value'=>'');
        unless ($treeitem eq '#') {
            $link_struct->createField('Type'=>FIELD_BYTE,'Label'=>'IsChild','Value'=>0);
        }
        push @$substruct_arr_ref,$link_struct;
    }
    else {  #we have to add the entire struct from our clipboard (not just a symbol)

        my $child_treeitem;                          # create empty linked structure(s)
        my $child_struct_arr_ref;
        if ($clipboard{'old lastbranch'}=~/E/) {
            $child_treeitem=add_entry($treeitem);
            $child_struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
        }
        elsif ($clipboard{'old lastbranch'}=~/R/) {
            $child_treeitem=add_reply($treeitem);
            $child_struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
        }
                                                     # now fill the new child_treeitem structure with clipboard
        my $child_lastbranch=(split /#/,$child_treeitem)[-1];
        $child_lastbranch =~/(\d+)/;
        my $child_struct_ix=$1;
        $$child_struct_arr_ref[$child_struct_ix]=$clipboard{'struct'};

        # because we have a struct in memory and not just a symbol, we should change that in case
        # the user wants to paste a copy of this item
        $clipboard{'symbol'}=1;
        $clipboard{'old treeitem'}=$child_treeitem;
        $clipboard{'old lastbranch'}=(split /#/,$child_treeitem)[-1];
        my $edit_menu=$menubar->entrycget('Edit',-menu);
        $edit_menu->entryconfigure('Paste Node As Copy',-state=>'normal');
    }
    gff_updated();
    refresh_tree();
}

sub splash{
    my $splash=$mw->Dialog(-text=>"DLGEditor v$version\n by tk102\n$compilation_date\nzxcvbnm6012\@yahoo.com",-title=>'About DLGEditor');
    $splash->Show();
}

#======================
# Internal functions
#======================


sub disable_menus{  #this will remove the Edit and View menus, and cripple the File menu
   my $file_menu=$menubar->entrycget('File',-menu);
   $file_menu->entryconfigure('Revert',-state=>'disabled');
   $file_menu->entryconfigure('Save',-state=>'disabled');
   $file_menu->entryconfigure('Save As...',-state=>'disabled');
   $file_menu->entryconfigure('Close',-state=>'disabled');
   my $second_menu=$menubar->entrycget('1',-label);
   if  ($second_menu =~ /Edit/) {
   $menubar->delete('Edit');
   $menubar->delete('View');
   }
}
sub enable_menus{
    #this will restore (but not enable) the Edit/View menus and enable File:Close
    return unless (ref $gff eq "Bioware::GFF");
    my $file_menu=$menubar->entrycget('File',-menu);
    #$file_menu->entryconfigure('Save',-state=>'normal');
    $file_menu->entryconfigure('Save As...',-state=>'normal');
    $file_menu->entryconfigure('Close',-state=>'normal');
    my $second_menu=$menubar->entrycget('1',-label);
    unless ($second_menu =~ /Edit/) {
        $menubar->delete('Help');
        $menubar->cascade(-label=>'~Edit',-menuitems=>mnuEdit,-tearoff=>0);
        $menubar->cascade(-label=>'~View',-menuitems=>mnuView,-tearoff=>0);
        $menubar->cascade(-label=>'~Help',-menuitems=>mnuHelp,-tearoff=>0);
    }

    my $edit_menu=$menubar->entrycget('Edit',-menu);
    $edit_menu->entryconfigure('Add New Entry',-state=>'normal');
    $edit_menu->entryconfigure('Add New Reply',-state=>'normal');
    $edit_menu->entryconfigure('Break Link With Parent',-state=>'normal');
    $edit_menu->entryconfigure('Delete Node Completely',-state=>'normal');
    $edit_menu->entryconfigure('Cut Node',-state=>'normal');
    $edit_menu->entryconfigure('Copy Node',-state=>'normal');
}
sub gff_updated { #called after user clicks Apply button
    #Update the Save and SaveAs Menus and set our 'GFFdirty' flag;
    my $file_menu=$menubar->entrycget('File',-menu);
    $file_menu->entryconfigure('Revert',-state=>'normal');
    $file_menu->entryconfigure('Save',-state=>'normal');
    $file_menu->entryconfigure('Save As...',-state=>'normal');
    $gff_dirty_flag=1;  #means GFF, the object != GFF, the file
}
sub last_chance_to_save {
    #used to give the user one last chance to save their work before
    #closing the current GFF object
    if ($gff_dirty_flag) {
        my $d = $mw->DialogBox(-title => "Save Changes?",-buttons => ["Save", "Save As...","Close"]);
        $d->add('Label', -text=>"Do you wish to save changes to $current_dialog?")->pack;
        my $answer=$d->Show();
        if ($answer eq 'Save') {
            save_dialog();
        }
        elsif ($answer eq 'Save As...') {
            save_as_dialog();
        }
    }
}
sub refresh_tree { #this subroutine redraws the tree with the GFF in memory
    return unless (ref $gff eq 'Bioware::GFF');
    return unless ($gff->{'sig'} eq 'DLG ');
    clear_tree();
    populate_tree();
}
sub refresh_tree_gentle { # this subroutine steps through each treeitem and effects the -text of each
                          # the effect is that leafs are not opened or closed automatically
    my $treeitem='#';
    my $entrylist_ix=$gff->{Main}->get_field_ix_by_label('EntryList');
    my $replylist_ix=$gff->{Main}->get_field_ix_by_label('ReplyList');
    my $entrylist_arr_ref=$gff->{Main}{Fields}[$entrylist_ix]{Value};
    my $replylist_arr_ref=$gff->{Main}{Fields}[$replylist_ix]{Value};
    while ($treeitem=$tree->infoNext($treeitem)) {
        my $struct;
        my $text;
        my $lastbranch = (split /#/,$treeitem)[-1];
        $lastbranch=~/(\d+)/;
        my $index=$1;
        my $end_dialog_flag=0;
        if ($lastbranch =~ /E/) {
            $struct=$$entrylist_arr_ref[$index];
            $text=get_entry_text_by_index($index);
            if ((my $oldtext=$tree->entrycget($treeitem,-text)) =~ /\(already listed\)/) {
                $text .= ' (already listed)';
            }
            elsif ((scalar ($tree->infoChildren($treeitem)))==0) {
                $end_dialog_flag=1;
            }
        }
        elsif ($lastbranch =~ /R/) {
            $struct=$$replylist_arr_ref[$index];
            $text=get_reply_text_by_index($index);
            if ((my $oldtext=$tree->entrycget($treeitem,-text)) =~ /\(already listed\)/) {
                $text .= ' (already listed)';
            }
            elsif ((scalar ($tree->infoChildren($treeitem)))==0) {
                $end_dialog_flag=1;
            }

        }
        if ($text eq '') {
            $text ='(continue)'
        }
        if ($lastbranch =~ /O/) {
                $text="(orphan) $text";
        }
        if ($view_scripts) {
            my $script=$struct->{Fields}[$struct->get_field_ix_by_label('Script')]{Value};
            $text .=" <$script>";
        }
        if ($view_conditionals) {
            my ($parent_type, $parent_struct, $parent_struct_index, $parent_substruct, $parent_substruct_index)=get_parent_child_link_structs($treeitem);
            my $script=$parent_substruct->{Fields}[$parent_substruct->get_field_ix_by_label('Active')]{Value};
            $text ="<$script> $text";
        }
        if ($view_indices) {
            if ($lastbranch =~ /E/) {
                $text="[E$index] $text";
            }
            elsif ($lastbranch =~ /R/) {
                $text="[R$index] $text";
            }
        }
        if ($end_dialog_flag) {
            $text .= " [End Dialog]";
        }
        $tree->entryconfigure($treeitem, -text=>$text);
    }
}
sub tree_click {
    my $treeitem=shift;

    #no re-entry on button-click, button-release
    return if ($treeitem eq $old_selection);
    $old_selection=$treeitem;

    my $lastbranch_name=(split /#/,$treeitem)[-1];
    if ($lastbranch_name =~/E/) {
        #Entry was clicked
        show_detail_widgets($treeitem,$lastbranch_name,'E'); #redundant info, but I'm lazy
    }
    elsif ($lastbranch_name =~/R/) {
        #Reply was clicked
        show_detail_widgets($treeitem,$lastbranch_name,'R'); #redundant info, but I'm lazy
    }
    elsif ($lastbranch_name eq '') {
        #root was clicked
        show_general_detail_widgets();
    }
}
sub tree_rclick {
    #simulate a left-click
    my $w=shift;
    my $ev = $w->XEvent();
    $tree->Button1($ev);
    $tree->ButtonRelease1($ev);
    my $treeitem=($tree->info('selection'))[0]; #get the item name selected
    #now perform right-click actions...

    #create context menu
    my $context_menu = $mw->Menu(-tearoff  => 0);
    if ($treeitem eq '') {     # empty tree
        $context_menu->add('command',-label=>'New',-command=>sub{ create_new_dialog() });
        $context_menu->add('command',-label=>'Open',-command=>sub{ open_dialog() });
    }
    else {
        if ($treeitem eq '#') { # root selected
            $context_menu->add('command',-label=>'New',-command=>sub{ create_new_dialog() });
            $context_menu->add('command',-label=>'Open',-command=>sub{ open_dialog() });
            my $state=$gff_dirty_flag ? 'normal' : 'disabled';
            $context_menu->add('command',-label=>'Revert',-command=>sub{ open_dialog_from_arg($current_dialog) },-state=>$state);
            $context_menu->add('command',-label=>'Save',-command=>sub{ save_dialog() },-state=>$state);
            $context_menu->add('command',-label=>'Save As...',-command=>sub{ save_as_dialog() },-state=>$state);
            $context_menu->add('command',-label=>'Close',-command=>sub{ close_dialog() });
            $context_menu->add('separator');
            $context_menu->add('command',-label=>'Add New Entry',-command=>sub { add_entry(undef) });
            $context_menu->add('command',-label=>'Add New Reply',-command=>sub { add_reply(undef) });
            $context_menu->add('separator');
            my $edit_menu=$menubar->entrycget('Edit',-menu);
            $state=$edit_menu->entrycget('Paste Node As Copy',-state);
            $context_menu->add('command',-label=>'Paste Node As Copy', -command=>sub { paste_node('copy') }, -state=>$state);
            $state=$edit_menu->entrycget('Paste Node As New',-state);
            $context_menu->add('command',-label=>'Paste Node As New', -command=>sub { paste_node('new') }, -state=>$state);
        }
        else {                   # treeitem selected
            my $text=$tree->entrycget($treeitem,-text);
            if ($text =~ /\(already listed\)/) {
                $context_menu->add('command',-label=>'Jump To Original',-accelerator=>'Ctrl+J',-command=>sub { jump_to_original($treeitem) });
                $context_menu->add('separator');
            }
            $context_menu->add('command',-label=>'Add New Entry',-command=>sub { add_entry(undef) });
            $context_menu->add('command',-label=>'Add New Reply',-command=>sub { add_reply(undef) });
            $context_menu->add('separator');
            my $state=$treeitem =~ /O/ ? 'disabled' : 'normal';
            $context_menu->add('command',-label=>'Break Link With Parent',-command=>sub { break_link_with_parent(); refresh_tree(); } , -state=>$state);
            $context_menu->add('command',-label=>'Delete Node Completely',-command=>sub { delete_node() });
            $context_menu->add('separator');
            $context_menu->add('command',-label=>'Cut Node',-command=>sub { cut_node() });
            $context_menu->add('command',-label=>'Copy Node',-command=>sub { copy_node() });
            my $edit_menu=$menubar->entrycget('Edit',-menu);
            $state=$edit_menu->entrycget('Paste Node As Copy',-state);
            $context_menu->add('command',-label=>'Paste Node As Copy', -command=>sub { paste_node('copy') }, -state=>$state);
            $state=$edit_menu->entrycget('Paste Node As New',-state);
            $context_menu->add('command',-label=>'Paste Node As New', -command=>sub { paste_node('new') }, -state=>$state);
        }
        $context_menu->add('separator');
        $context_menu->add('checkbutton',-label=>'Show Conditionals in Tree',-variable=>\$view_conditionals,-command=>sub { refresh_tree_gentle() });
        $context_menu->add('checkbutton',-label=>'Show Scripts in Tree',-variable=>\$view_scripts,-command=>sub { refresh_tree_gentle() });
        $context_menu->add('checkbutton',-label=>'Show Indices in Tree',-variable=>\$view_indices,-command=>sub { refresh_tree_gentle() });
        $context_menu->add('checkbutton',-label=>'Show Orphaned Replies/Entries',-variable=>\$view_orphans,-command=> sub { refresh_tree() });
        $context_menu->add('separator');
        $context_menu->add('command',-label=>'Expand All', -command=>sub { expand_tree();});
        $context_menu->add('command',-label=>'Fold All', -command=>sub { fold_tree();});
    }
    #show popup at cursor
    $context_menu->Popup(
    -popover  => 'cursor',
    -popanchor => 'nw');


}

sub clear_tree {
    my $treeitem = '#';
    if ($tree->infoExists('#')) {
    while ($treeitem=$tree->infoNext($treeitem)) {
        $tree_memory{$treeitem}=$tree->infoHidden($treeitem);
    }}
    $tree->delete('all');
    %entry_ixs=();
    %reply_ixs=();
}
sub populate_tree {
    #this subroutine will populate the tree widget with the GFF info
    #refresh_tree_gentle will fill in human readable
    if ($current_dialog) {
        $tree->add('#',-text=>"$current_dialog",-itemtype=>'text',-style=>$rootstyle);
    }
    else {
        $tree->add('#',-text=>"New Dialog",-itemtype=>'text',-style=>$rootstyle);
    }
    # loop through starting list
    my $startinglist_ix=$gff->{Main}->get_field_ix_by_label('StartingList');
    my $entrylist_ix=$gff->{Main}->get_field_ix_by_label('EntryList');
    #my $startinglist_arr_ref=$gff->{Main}{Fields}[$startinglist_ix]{Value};
    my $i=0;
    for (my $i=0; $i< scalar @{$gff->{Main}{Fields}[$startinglist_ix]{Value}}; $i++) {
        my $startinglist_struct=$gff->{Main}{Fields}[$startinglist_ix]{Value}[$i];
        my $entryfield_ix=$startinglist_struct->get_field_ix_by_label('Index');
        my $index=$startinglist_struct->{'Fields'}[$entryfield_ix]{'Value'};
        #check validity of startinglist entry here (not in populate_children)
        my $temp_entry_struct=$gff->{Main}{Fields}[$entrylist_ix]{Value}[$index];
        if (ref $temp_entry_struct eq 'Bioware::GFF::Struct') {
            my $entry_ix;
            $entry_ix=$index; #use $index for memory, and use $entry_ix for treeitem
            while ($tree->infoExists("#E$entry_ix")) {
                $entry_ix.=' ';
            }
            my $entry_text;                                 #we need to do this here because
            if (exists $entry_ixs{$index}) {                #refresh_tree_gentle looks for it
                $entry_text = '(already listed)';
            }
            $tree->add("#E$entry_ix",-itemtype=>'text',-text=>$entry_text,-style=>$entrystyle);
            $entry_ixs{$index}=1;
            populate_children("#E$entry_ix");
            return unless $gff;
        }
        else {
            my @buttons=("Open in GFF Editor","Open in DLGEdit","Close DLG");
            my $d = $mw->DialogBox(-title => "Error in parsing", -buttons => [@buttons]);
            for my $b (@buttons) {
               my $bb="B_".$b;
               my $btn=$d->Subwidget($bb);
               $btn->configure(-width=>length $b)
            }

            $d->add('Label', -text=>"There is an error in StartingList$i which tried to reference E$index which does not exist.\n"
                  ."You will need to fix this in before DLGEditor can display the dialog tree properly.")->pack;
            my $btn=$d->Show();
            if ($btn eq "Open in GFF Editor") {
               launch_gffeditor();
            }
            elsif ($btn eq "Open in DLGEdit") {
               launch_dlgedit();
            }
            close_dialog;
            return;
        }
    }
    # now, if we had an error in our startinglist, then the error will not be present in our new array
    # so we should update the gff and start over
    if ($view_orphans) {
        # find orphans in entry list --
        # open up EntryList and count number entry structs (n)
        # then count from 0 to n and see if in %entry_ixs

        my $the_entrylist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
        my $total_entry_count =scalar @$the_entrylist_arr_ref;
        for (my $i=0; $i<$total_entry_count; $i++) {
            unless ($entry_ixs{$i}) {    #need to create this entry orphan
                $tree->add("#OE$i",-itemtype=>'text',-text=>'',-style=>$orphan_entrystyle,-at=>0);
                populate_children("#OE$i");
                return unless $gff;
            }
        }
        # find orphans in reply list --
        # open up ReplyList and count number entry structs (n)
        # then count from 0 to n and see if in %reply_ixs

        my $the_replylist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
        my $total_reply_count =scalar @$the_replylist_arr_ref;
        for (my $i=0; $i<$total_reply_count; $i++) {
            unless ($reply_ixs{$i}) {    #need to create this reply orphan
                $tree->add("#OR$i",-itemtype=>'text',-text=>'',-style=>$orphan_replystyle,-at=>0);
                populate_children("#OR$i");
                return unless $gff;

            }
        }
    }
    #my @rev_sorted_keys=reverse (keys %tree_memory)
    for my $ti (keys %tree_memory) {
        if ($tree->infoExists($ti)) {
                $tree->hide('entry',$ti) if ($tree_memory{$ti});
        }
    }
    $tree->autosetmode();
    refresh_tree_gentle();
}
sub populate_children ($) {
    my $treeitem=shift;
    my $lastbranch=(split /#/,$treeitem)[-1];
    if ($lastbranch =~ /E/) {                                                     # search for replieslist for this entry
        my $this_item=(split /E/,$lastbranch)[-1];
        my $entrylist_ix=$gff->{Main}->get_field_ix_by_label('EntryList');
        my $entry_struct=$gff->{Main}{Fields}[$entrylist_ix]{Value}[$this_item];
        my $replieslist_ix;
        eval {$replieslist_ix=$entry_struct->get_field_ix_by_label('RepliesList');};
        if ($@) {
            my $the_culprit;
            $the_culprit=(split /#/,$treeitem)[-2];
            my @buttons=("Remove reference and continue","Open in GFF Editor","Open in DLGEdit");
            my $d = $mw->DialogBox(-title => "Error in parsing", -buttons => [@buttons]);
            for my $b (@buttons) {
               my $bb="B_".$b;
               my $btn=$d->Subwidget($bb);
               $btn->configure(-width=>length $b)
            }
            $d->add('Label', -text=>"There is an error in $the_culprit which tried to reference $lastbranch which does not exist.")->pack;
            my $btn=$d->Show();
            if ($btn eq "Remove reference and continue") {
               break_link_with_parent($treeitem);
            }
            elsif ($btn eq "Open in GFF Editor") {
               launch_gffeditor();
               close_dialog();
            }
            elsif ($btn eq "Open in DLGEdit") {
               launch_dlgedit();
               close_dialog();
            }
            return;
        };
        my $replieslist_arr_ref=$entry_struct->{Fields}[$replieslist_ix]{Value};
        if (scalar @$replieslist_arr_ref == 0) {
            my $old_treetext=$tree->entrycget($treeitem,-text);
            $tree->entryconfigure($treeitem,-text=>"$old_treetext [End Dialog]");
        }
        for my $replieslist_struct (@$replieslist_arr_ref) {                      # let's look at the children (Replies) of this Entry....
            my $index_ix=$replieslist_struct->get_field_ix_by_label('Index');
            my $index=$replieslist_struct->{Fields}[$index_ix]{Value};
            my $text=get_reply_text_by_index($index);
            unless ($text) {$text = '(continue)'}
            if ($view_conditionals) {
                my $active=$replieslist_struct->{Fields}[$replieslist_struct->get_field_ix_by_label('Active')]{Value};
                $text = '<' . $active . '> ' . $text;
            }
            if ($view_scripts) {
                my $replylist_ix=$gff->{Main}->get_field_ix_by_label('ReplyList');
                my $reply_struct=$gff->{Main}{Fields}[$replylist_ix]{Value}[$index];
                my $script=$reply_struct->{Fields}[$reply_struct->get_field_ix_by_label('Script')]{Value};
                $text .= ' <' . $script . '>';
            }
            if ($view_indices) {
                $text = '[R' . $index . '] '.$text;
            }


            my $new_treeitem="$treeitem#R$index";
            while ($tree->infoExists("$new_treeitem")) {
                $new_treeitem.=' ';  #this bit is for the case when two replies of the same index are listed under an entry
            }
            if (exists $reply_ixs{$index}) {
                $tree->add($new_treeitem,-itemtype=>'text',-text=>"$text (already listed)",-style=>$replystyle);
            }
            else {
                $tree->add($new_treeitem,-itemtype=>'text',-text=>$text,-style=>$replystyle);
                $reply_ixs{$index}=1;
                populate_children("$treeitem#R$index");
                return unless $gff;
            }
        }
    }
    elsif ($lastbranch =~ /R/) {                                                # search for entrieslist for this reply
        my $this_item=(split /R/,$lastbranch)[-1];
        my $replylist_ix=$gff->{Main}->get_field_ix_by_label('ReplyList');
        my $reply_struct=$gff->{Main}{Fields}[$replylist_ix]{Value}[$this_item];
        my $entrieslist_ix;
        eval {$entrieslist_ix=$reply_struct->get_field_ix_by_label('EntriesList');};
        if ($@) {
          my $the_culprit=(split /#/,$treeitem)[-2];
            my @buttons=("Remove reference and continue","Open in GFF Editor","Open in DLGEdit");
            my $d = $mw->DialogBox(-title => "Error in parsing", -buttons => [@buttons]);
            for my $b (@buttons) {
               my $bb="B_".$b;
               my $btn=$d->Subwidget($bb);
               $btn->configure(-width=>length $b)
            }
            $d->add('Label', -text=>"There is an error in $the_culprit which tried to reference $lastbranch which does not exist.")->pack;
            my $btn=$d->Show();
            if ($btn eq "Remove reference and continue") {
               break_link_with_parent($treeitem);
            }
            elsif ($btn eq "Open in GFF Editor") {
               launch_gffeditor();
               close_dialog();
            }
            elsif ($btn eq "Open in DLGEdit") {
               launch_dlgedit();
               close_dialog();
            }
            return;

        };
        my $entrieslist_arr_ref=$reply_struct->{Fields}[$entrieslist_ix]{Value};
        for my $entrieslist_struct (@$entrieslist_arr_ref) {                    # let's look at the children (Entries) of this Reply
            my $index_ix=$entrieslist_struct->get_field_ix_by_label('Index');
            my $index=$entrieslist_struct->{Fields}[$index_ix]{Value};
            my $text=get_entry_text_by_index($index);
            unless ($text) {$text = '(continue)'}
            if ($view_conditionals) {
                my $active=$entrieslist_struct->{Fields}[$entrieslist_struct->get_field_ix_by_label('Active')]{Value};
                if ($active) {
                    $text = '<' . $active . '> ' . $text;
                }
            }
            if ($view_scripts) {
                my $entrylist_ix=$gff->{Main}->get_field_ix_by_label('EntryList');
                my $entry_struct=$gff->{Main}{Fields}[$entrylist_ix]{Value}[$index];
                my $script=$entry_struct->{Fields}[$entry_struct->get_field_ix_by_label('Script')]{Value};
                if ($script) {
                    $text .= ' <' . $script . '>';
                }
            }
            if ($view_indices) {
                $text = '[E' . $index . '] '.$text;
            }

            my $new_treeitem="$treeitem#E$index";
            while ($tree->infoExists("$new_treeitem")) {
                $new_treeitem.=' ';  #this bit is for the case when two entries of the same index are listed under a reply
                                    #which could happen if they use two different Active fields
            }
            if (exists $entry_ixs{$index}) {
                $tree->add($new_treeitem,-itemtype=>'text',-text=>"$text (already listed)",-style=>$entrystyle);
            }
            else {
                $tree->add($new_treeitem,-itemtype=>'text',-text=>$text,-style=>$entrystyle);
                $entry_ixs{$index}=1;
                populate_children("$treeitem#E$index");
                return unless $gff;
            }
        }
    }
}
sub get_reply_text_by_index ($) {
    my $reply_index=shift;
    return unless $gff;
    my $reply_field_ix=$gff->{Main}->get_field_ix_by_label('ReplyList');
    my $reply_struct=$gff->{Main}{Fields}[$reply_field_ix]{Value}[$reply_index];
    unless (defined $reply_struct) {return};
    my $text_field_ix=$reply_struct->get_field_ix_by_label('Text');
    my $strref=$reply_struct->{Fields}[$text_field_ix]{Value}{StringRef};
    my $reply_string;
    if ($strref>-1) {
        $reply_string=string_from_resref($path_to_dialog_dot_tlk,$strref);
    } else {
        $reply_string=$reply_struct->{Fields}[$text_field_ix]{Value}{Substrings}[0]{Value};
    }
    return $reply_string;
}
sub get_entry_text_by_index ($) {
    my $entry_index=shift;
    return unless $gff;
    my $entry_field_ix=$gff->{Main}->get_field_ix_by_label('EntryList');
    my $entry_struct=$gff->{Main}{Fields}[$entry_field_ix]{Value}[$entry_index];
    unless(defined $entry_struct) {return};
    my $text_field_ix=$entry_struct->get_field_ix_by_label('Text');
    my $strref=$entry_struct->{Fields}[$text_field_ix]{Value}{StringRef};
    my $entry_string;
    if ($strref>-1) {
        $entry_string=string_from_resref($path_to_dialog_dot_tlk,$strref);
    } else {
        $entry_string=$entry_struct->{Fields}[$text_field_ix]{Value}{Substrings}[0]{Value};
    }
    return $entry_string;
}
sub destroy_old_widgets {
    for my $widge (@spawned_widgets) {   #unspawn old widgets
        $widge->destroy if Tk::Exists($widge);
    }
    @spawned_widgets=();

}
sub show_detail_widgets {

    my ($treeitem, $lastbranch_name, $e_or_r)=@_;
    my @colrow=(0,0);
    destroy_old_widgets();

    #figure out what we're dealing with
    my $struct_arr_ref;
    if ($e_or_r eq 'E') {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
    }
    elsif ($e_or_r eq 'R') {
        $struct_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
    }
    $lastbranch_name=~/(\d+)/;
    my $struct_index=$1;
    my $struct=$$struct_arr_ref[$struct_index];

    my ($parent_type,$parent_struct,$parent_struct_index, $parent_substruct, $parent_substruct_index)=get_parent_child_link_structs($treeitem);

    my $txtAvailabilityScript_var;
    my $script_ix;
    unless ($parent_type eq 'O') {
        $script_ix=$parent_substruct->get_field_ix_by_label('Active');
        $txtAvailabilityScript_var=$parent_substruct->{Fields}[$script_ix]{Value};
    }

    #spawn widgets
    #~~~~~~~~~~~~~

    # Entry/Reply/Start number

    my $lblName;
    if ($e_or_r eq 'E') {
        $lblName="Entry"
    } elsif ($e_or_r eq 'R') {
        $lblName="Reply"
    }
    $lblName .= " $struct_index (Parent: ";
    if ($parent_type eq 'S') {
        $lblName .="StartingList $parent_substruct_index)";
    } elsif ($parent_type eq 'E') {
        $lblName .="Entry $parent_struct_index)";
    } elsif ($parent_type eq 'R') {
        $lblName .="Reply $parent_struct_index)";
    } elsif ($parent_type eq 'O') {
        $lblName .="none)";
    }
    $lblName=$lower_frame->Label(-text=>$lblName,-font=>['Futura','12','bold']
                                #)->grid(-sticky=>'ne',-column=>0,-row=>$colrow[0]);
                                )->place(-x=>0,-y=>0,-anchor=>'nw');
    $colrow[0]++;

    push @spawned_widgets, $lblName;

    #Speaker/Listener

    my $lblSpeaker;
    if ($e_or_r eq 'E') {
        $lblSpeaker="Speaker";
    } elsif ($e_or_r eq 'R') {
        $lblSpeaker="Listener";
    }
    my $txtSpeaker_var;
    if (defined $struct->get_field_ix_by_label($lblSpeaker)) {
      $txtSpeaker_var=$struct->{Fields}[$struct->get_field_ix_by_label($lblSpeaker)]{Value};
    }
    my $txtSpeaker=$lower_frame->LabEntry(-label=>$lblSpeaker,-labelPack=>[-side=>'left',-anchor=>'w'],
                                          -textvariable=>\$txtSpeaker_var,-background=>'white',-width=>16);
    $txtSpeaker->grid(-sticky=>'ne',-column=>0,-row=>$colrow[0]);
    $colrow[0]++;
    push @spawned_widgets,($txtSpeaker);

    #StrRef/Text

    my $text_ix=$struct->get_field_ix_by_label('Text');
    my $txtStrRef_var;
    if (defined $text_ix) {
       $txtStrRef_var=$struct->{Fields}[$text_ix]{Value}{StringRef};
    }
    my $txtText;
    my $txtText_var;
    $txtText=$lower_frame->Scrolled("Text",-scrollbars=>'oe',-background=>'white',-width=>40,-wrap=>'word',height=>3);
    if ($txtStrRef_var == -1) {
        $txtText_var=$struct->{Fields}[$text_ix]{Value}{Substrings}[0]{Value} ; # text is now a string
        $txtText->delete('1.0','end');
        $txtText->insert('end',$txtText_var);
    }
    else {
        $txtText_var=string_from_resref($path_to_dialog_dot_tlk,$txtStrRef_var);
        $txtText->delete('1.0','end');
        $txtText->insert('end',$txtText_var);
        $txtText->configure(-state=>'disabled', -background=>'light gray');
    }
    my $txtStrRef=$lower_frame->LabEntry(-textvariable=>\$txtStrRef_var,
                                      -background=>'white',
                                      -width=>14,
                                      -label=>'StrRef',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(-?\d*\.?\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]==-1) {
                                                $txtText->configure(-state=>'normal',-background=>'white');
                                            } elsif ($_[0]>-1) {
                                                $txtText_var=string_from_resref($path_to_dialog_dot_tlk,$_[0]);
                                                $txtText->configure(-state=>'normal');
                                                $txtText->delete('1.0','end');
                                                $txtText->insert("end",$txtText_var);
                                                $txtText->configure(-state=>'disabled',-background=>'light gray');
                                            }
                                            return 1;
                                        } return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell});
    $txtStrRef->grid(-row=>1,-column=>1,-sticky=>'se');
    $colrow[1]=2;
    $txtText->grid(-row=>$colrow[1],-rowspan=>3,-column=>1,-sticky=>'ne');
    $colrow[1]+=3;
    push @spawned_widgets,($txtText,$txtStrRef);

    #Availability Script (from parent)
    my $txtAvailabilityScript=$lower_frame->LabEntry(-textvariable=>\$txtAvailabilityScript_var,-background=>'white',
                                           -label=>'Script that deterimines availability',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtAvailabilityScript->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtAvailabilityScript;

    #Script to fire
    my $txtScriptToFire_var;
    if (defined $struct->get_field_ix_by_label('Script')) {
      $txtScriptToFire_var=$struct->{Fields}[$struct->get_field_ix_by_label('Script')]{Value};
    }
    my $txtScriptToFire=$lower_frame->LabEntry(-textvariable=>\$txtScriptToFire_var,-background=>'white',
                                           -label=>'Script that fires when spoken',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtScriptToFire->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtScriptToFire;

    #Vo_resref

    my $txtVOResRef_var;
    if (defined ($struct->get_field_ix_by_label('VO_ResRef'))) {
      $txtVOResRef_var=$struct->{Fields}[$struct->get_field_ix_by_label('VO_ResRef')]{Value};
    }
    my $txtVOResRef=$lower_frame->LabEntry(-textvariable=>\$txtVOResRef_var,-background=>'white',
                                           -label=>'VO_ResRef',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtVOResRef->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtVOResRef;

    #Sound
    my $txtSound_var;
    if (defined ($struct->get_field_ix_by_label('Sound'))) {
      $txtSound_var=$struct->{Fields}[$struct->get_field_ix_by_label('Sound')]{Value};
    }
    my $txtSound=$lower_frame->LabEntry(-textvariable=>\$txtSound_var,-background=>'white',
                                           -label=>'Sound',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtSound->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtSound;

    #Sound Exists
    my $fraSoundExists=$lower_frame->Frame();

    my $chkSoundExists_var;
    if (defined ($struct->get_field_ix_by_label('SoundExists'))) {
      $chkSoundExists_var=$struct->{Fields}[$struct->get_field_ix_by_label('SoundExists')]{Value};
    }
    $chkSoundExists_var=($chkSoundExists_var>0);
    my $chkSoundExists=$fraSoundExists->Checkbutton(-variable=>\$chkSoundExists_var);
    my $lblSoundExists=$fraSoundExists->Label(-text=>'Sound Exists')->grid($chkSoundExists);

    $fraSoundExists->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets, ($fraSoundExists,$chkSoundExists,$lblSoundExists);

    #Animation List
    my $animlist_frame=$lower_frame->Frame();

    my $lbl_anim_avail=$animlist_frame->Label(-text=>'Available Animations (drag to add)');
    my $lbl_anim_current=$animlist_frame->Label(-text=>'Current Animations [Participant]');
    $lbl_anim_current->grid($lbl_anim_avail,-padx=>10);

    my $animation_list_current=$animlist_frame->Scrolled("Listbox",-scrollbars=>'oe',
                                                      -height=>'4',-width=>'25',
                                                      -background=>'white');
    my $animation_list_source=$animlist_frame->Scrolled("Listbox",-scrollbars=>'oe',
                                                     -height=>'4',-width=>'25',
                                                     -background=>'white');
    $animation_list_current->bind('<Button-3>'=>sub { animation_list_rclick($animation_list_current,$animation_list_source) });
    $animation_list_current->grid($animation_list_source);

    my %animation_desc2num=reverse %animation_num2desc;
    my @animations=sort (keys %animation_desc2num);
    $animation_list_source->insert('end',@animations);
    $animlist_frame->grid(-sticky=>'news',-column=>1,-row=>$colrow[1],-rowspan=>5);
    $colrow[1]+=5;
    my $dnd_token_add;
    $dnd_token_add=$animation_list_source->DragDrop(
                                                -event     => '<B1-Motion>',
                                                -sitetypes => ['Local'],
                                                -startcommand => sub {StartDrag($dnd_token_add)}
                                                );
    $animation_list_current->DropSite(
                                      -droptypes=>['Local'],
                                      -dropcommand=>[ \&DropAdd, $animation_list_current, $dnd_token_add ]
                                      );

    $mw->bind('<KeyRelease-Delete>'=>sub {
        if ($animation_list_current->curselection) {
        $animation_list_current->delete($animation_list_current->curselection)}
    });

    my $animlist_arr_ref;
    if (defined($struct->get_field_ix_by_label('AnimList'))) {
      $animlist_arr_ref=$struct->{Fields}[$struct->get_field_ix_by_label('AnimList')]{Value};
      for my $anim_struct (@$animlist_arr_ref) {
        my $this_animation;
        if (defined ($anim_struct->get_field_ix_by_label('Animation'))) {
         $this_animation=$anim_struct->{Fields}[$anim_struct->get_field_ix_by_label('Animation')]{Value};
        }
        my $this_participant;
        if (defined ($anim_struct->get_field_ix_by_label('Participant'))) {
         $this_participant=$anim_struct->{Fields}[$anim_struct->get_field_ix_by_label('Participant')]{Value};
        }
        $animation_list_current->insert('end',"$animation_num2desc{$this_animation} [$this_participant]");
      }
    }
    # options at this point:
    # 1. add a new "garbage can" target for dropsite
    # 2. limit animation lists to 1 and use dropdown list to select + participant LabEntry
    # 3. use Delete button to remove items from list
    # 4. context menu to add/delete/modify animlist
    push @spawned_widgets, ($animlist_frame,$lbl_anim_avail,$lbl_anim_current,$animation_list_current,$animation_list_source);#$dnd_token_remove,$dnd_token_add);

    #Delay
    my $txtDelay_var;
    if (defined($struct->get_field_ix_by_label('Delay'))) {
      $txtDelay_var=$struct->{Fields}[$struct->get_field_ix_by_label('Delay')]{Value};
    }
    $txtDelay_var=unpack('l',pack('L',$txtDelay_var));
    my $txtDelay=$lower_frame->LabEntry(-textvariable=>\$txtDelay_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'Delay',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(-?\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=-1) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets, $txtDelay;
    #Wait Flags

    my $txtWaitFlags_var;
    if (defined ($struct->get_field_ix_by_label('WaitFlags'))){
      $txtWaitFlags_var=$struct->{Fields}[$struct->get_field_ix_by_label('WaitFlags')]{Value};
    }
    my $txtWaitFlags=$lower_frame->LabEntry(-textvariable=>\$txtWaitFlags_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'WaitFlags',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(-?\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=-1) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets, $txtWaitFlags;

    #Camera Angle

    my $txtCameraAngle_var;
    if (defined($struct->get_field_ix_by_label('CameraAngle'))) {
      $txtCameraAngle_var=$struct->{Fields}[$struct->get_field_ix_by_label('CameraAngle')]{Value};
    }
    my $txtCameraAngle=$lower_frame->LabEntry(-textvariable=>\$txtCameraAngle_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'CameraAngle',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=0) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid(-sticky=>'ne',-row=>$colrow[0],-column=>0);
    $colrow[0]++;
    push @spawned_widgets, $txtCameraAngle;



    #FadeType
    my $txtFadeType_var;
    if (defined($struct->get_field_ix_by_label('FadeType'))) {
      $txtFadeType_var=$struct->{Fields}[$struct->get_field_ix_by_label('FadeType')]{Value};
    }
    my $txtFadeType=$lower_frame->LabEntry(-textvariable=>\$txtFadeType_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'FadeType',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=0) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtFadeType;

    #Quest
    my $txtQuest_var;
    if (defined($struct->get_field_ix_by_label('Quest'))) {
      $txtQuest_var=$struct->{Fields}[$struct->get_field_ix_by_label('Quest')]{Value};
    }
    my $txtQuest=$lower_frame->LabEntry(-textvariable=>\$txtQuest_var,-background=>'white',
                                           -label=>'Quest',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtQuest->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtQuest;

    #PlotIndex
    my $txtPlotIndex_var;
    if (defined($struct->get_field_ix_by_label('PlotIndex'))) {
      $txtPlotIndex_var=$struct->{Fields}[$struct->get_field_ix_by_label('PlotIndex')]{Value};
    }
    my $txtPlotIndex=$lower_frame->LabEntry(-textvariable=>\$txtPlotIndex_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'PlotIndex',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(-?\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=-1) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtPlotIndex;

    #PlotXPPercentage
    my $txtPlotXPPer_var;
    if (defined($struct->get_field_ix_by_label('PlotXPPercentage'))) {
      $txtPlotXPPer_var=$struct->{Fields}[$struct->get_field_ix_by_label('PlotXPPercentage')]{Value};
    }
    my $txtPlotXPPer=$lower_frame->LabEntry(-textvariable=>\$txtPlotXPPer_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'PlotXPPercentage',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(\d*\.?\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=0) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtPlotXPPer;
    #Comment
    my $txtComment_var;
    if (defined($struct->get_field_ix_by_label('Comment'))) {
      $txtComment_var=$struct->{Fields}[$struct->get_field_ix_by_label('Comment')]{Value};
    }
    my $txtComment=$lower_frame->LabEntry(-textvariable=>\$txtComment_var,-background=>'white',
                                           -label=>'Comment',-labelPack=>[-side=>'left',-anchor=>'w'],
                                         );
    #bindDump $txtComment;

    $txtComment->grid(-row=>$colrow[0],-column=>0,-sticky=>'news',-columnspan=>2);
    $colrow[0]++;
    push @spawned_widgets,$txtComment;

    #Apply Changes
    my $btnApply=$lower_frame->Button(-text=>'Apply Changes',
                                      -borderwidth=>2,
                                      -width=>25,
                                      -height=>2,
                                      -command=>sub { #okay, this one is kind of big.
        if ($txtStrRef_var==-1) {
            my $new_text=$txtText->get('1.0','end');
            #$new_text =~ s/\000//;
            chomp $new_text;
            if (defined($struct->get_field_ix_by_label('Text'))) {
               $struct->{Fields}[$struct->get_field_ix_by_label('Text')]{Value}{StringRef}=-1;
               if ($new_text eq '') {
                   $struct->{Fields}[$struct->get_field_ix_by_label('Text')]{Value}{Substrings}=[];
               }
               else {
                   $struct->{Fields}[$struct->get_field_ix_by_label('Text')]{Value}{Substrings}[0]{Value}=$new_text;
               }
            }
            else {
               $struct->createField('Type'=>FIELD_CEXOLOCSTRING,'Label'=>'Text'           ,'StringRef'=>-1,'Value'=>$new_text);
            }

        }
        else {
            if (defined($struct->get_field_ix_by_label('Text'))) {
               $struct->{Fields}[$struct->get_field_ix_by_label('Text')]{Value}{StringRef}=$txtStrRef_var;
               $struct->{Fields}[$struct->get_field_ix_by_label('Text')]{Value}{Substrings}=[];
            }
            else {
               $struct->createField('Type'=>FIELD_CEXOLOCSTRING,'Label'=>'Text'           ,'StringRef'=>$txtStrRef_var);
            }

         }
        $parent_substruct->{Fields}[$script_ix]{Value}=$txtAvailabilityScript_var;
        if (defined($struct->get_field_ix_by_label($lblSpeaker))) {
            $struct->{Fields}[$struct->get_field_ix_by_label($lblSpeaker)]{Value}=$txtSpeaker_var;
         } else {
            $struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>$lblSpeaker        ,'Value'=>$txtSpeaker_var);
         }

         if (defined($struct->get_field_ix_by_label('Script'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('Script')]{Value}=$txtScriptToFire_var;
         } else {
            $struct->createField('Type'=>FIELD_RESREF,       'Label'=>'Script'         ,'Value'=>$txtScriptToFire_var);
         }
         if (defined($struct->get_field_ix_by_label('VO_ResRef'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('VO_ResRef')]{Value}=$txtVOResRef_var;
         } else {
            $struct->createField('Type'=>FIELD_RESREF,       'Label'=>'VO_ResRef'      ,'Value'=>$txtVOResRef_var);
         }
         if (defined($struct->get_field_ix_by_label('Sound'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('Sound')]{Value}=$txtSound_var;
         } else {
            $struct->createField('Type'=>FIELD_RESREF,       'Label'=>'Sound'          ,'Value'=>$txtSound_var);
         }
        if  (defined($struct->get_field_ix_by_label('SoundExists'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('SoundExists')]{Value}=$chkSoundExists_var;
        } else {
            $struct->createField('Type'=>FIELD_BYTE,         'Label'=>'SoundExists'    ,'Value'=>$chkSoundExists_var);
        }
        if  (defined($struct->get_field_ix_by_label('Delay'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('Delay')]{Value}=$txtDelay_var;
        } else {
            $struct->createField('Type'=>FIELD_DWORD,        'Label'=>'Delay'          ,'Value'=>$txtDelay_var);
        }
        if  (defined($struct->get_field_ix_by_label('WaitFlags'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('WaitFlags')]{Value}=$txtWaitFlags_var;
        } else {
            $struct->createField('Type'=>FIELD_DWORD,        'Label'=>'WaitFlags'      ,'Value'=>$txtWaitFlags_var);
        }
        if  (defined($struct->get_field_ix_by_label('CameraAngle'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('CameraAngle')]{Value}=$txtCameraAngle_var;
        } else {
            $struct->createField('Type'=>FIELD_DWORD,        'Label'=>'CameraAngle'    ,'Value'=>$txtCameraAngle_var);
        }
        if  (defined($struct->get_field_ix_by_label('FadeType'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('FadeType')]{Value}=$txtFadeType_var;
        } else {
            $struct->createField('Type'=>FIELD_BYTE,         'Label'=>'FadeType'       ,'Value'=>$txtFadeType_var);
        }
        if  (defined($struct->get_field_ix_by_label('Quest'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('Quest')]{Value}=$txtQuest_var;
        } else {
             $struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Quest'          ,'Value'=>$txtQuest_var);
        }
        if  (defined($struct->get_field_ix_by_label('PlotIndex'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('PlotIndex')]{Value}=$txtPlotIndex_var;
        } else {
            $struct->createField('Type'=>FIELD_INT,          'Label'=>'PlotIndex'      ,'Value'=>$txtPlotIndex_var);
        }
        if  (defined($struct->get_field_ix_by_label('PlotXPPercentage'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('PlotXPPercentage')]{Value}=$txtPlotXPPer_var;
        } else {
            $struct->createField('Type'=>FIELD_FLOAT,        'Label'=>'PlotXPPercentage','Value'=>$txtPlotXPPer_var);
        }
        if  (defined($struct->get_field_ix_by_label('Comment'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('Comment')]{Value}=$txtComment_var;
        } else {
            $struct->createField('Type'=>FIELD_CEXOSTRING,   'Label'=>'Comment'        ,'Value'=>$txtComment_var);
        }

        my @gff_anim_array=();

        for my $animitem ($animation_list_current->get(0,'end')) {
            next unless $animitem=~/(\w+) \[(\w+)\]/;
            my $new_animation=$animation_desc2num{$1};
            my $new_participant=$2;
            my $new_anim=Bioware::GFF::Struct->new('ID'=>0);
            $new_anim->createField('Type'=>FIELD_CEXOSTRING,'Label'=>'Participant','Value'=>$new_participant);
            $new_anim->createField('Type'=>FIELD_WORD,'Label'=>'Animation','Value'=>$new_animation);
            push @gff_anim_array, $new_anim;
        }
        if  (defined($struct->get_field_ix_by_label('AnimList'))) {
            $struct->{Fields}[$struct->get_field_ix_by_label('AnimList')]{Value}=[@gff_anim_array];
        } else {
            $struct->createField('Type'=>FIELD_LIST,         'Label'=>'AnimList'       ,'Value'=>[@gff_anim_array]);
        }
        gff_updated();
        refresh_tree_gentle();
                                        })->grid(
                                                   -row=>$colrow[0]-3,
                                                   -column=>1,
                                                   -rowspan=>2);
   push @spawned_widgets,$btnApply;
}

sub print_keysym{ #for binding debug only...
    my ($w)=@_;
    my $e=$w->XEvent;
    my ($kt,$kd)=($e->K,$e->N);
    print "keysym=$kt,numeric=$kd\n";
}
sub bindDump { #for debugging only...
   my $w=shift;
   print "Binding information for $w\n";
   foreach my $tag ($w->bindtags) {
       printf "\n Binding tag '$tag' has these bindings:\n";
       foreach my $binding ($w->bind($tag)) {
           printf "  $binding\n";
       }
   }
}
sub StartDrag {
    my($token) = @_;


    my $w = $token->parent; # $w is the source listbox
    my $e = $w->XEvent;
    my $idx = $w->nearest($e->y); # get the listbox entry under cursor
    if (defined $idx) {
        # Configure the dnd token to show the listbox entry
        $token->configure(-text => $w->get($idx));
        # Show the token
        my($X, $Y) = ($e->X, $e->Y);
        $token->MoveToplevelWindow($X, $Y);
        $token->raise;
        $token->deiconify;
        $token->FindSite($X, $Y, $e);
    }
}
sub DropAdd {
    my($lb, $dnd_source)= @_;
    my $t=$dnd_source->cget(-text);
    my $d = $mw->DialogBox(-title => "Enter Participant", -buttons => ["OK", "Cancel"]);
    my $txtvar="OWNER";
    $d->add('LabEntry', -width=>16,-textvariable=>\$txtvar,-label=>"Choose Participant for new $t Animation (default=OWNER)",-background=>'white')->pack;
    my $btn=$d->Show();
    return if ($btn eq "Cancel");

    $lb->insert("end","$t [$txtvar]");
    $lb->see("end");
}

sub show_general_detail_widgets {
    return unless $gff;

    destroy_old_widgets();
    my @colrow=(0,0,0,1);

    #Spawn General Widgets
    #~~~~~~~~~~~~~~~~~~~~~

    #Caption label
    my $lblCaption=$lower_frame->Label(-text=>"General Properties",-font=>['Futura','12','bold']
                                     # )->grid('-',-column=>0,-row=>$colrow[0]);
                                      )->place(-x=>0,-y=>0,-anchor=>'nw');
    $colrow[0]++;
    push @spawned_widgets, $lblCaption;



    #End Conversation Script
    my $txtEndConversationScript_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EndConversation')]{Value};
    my $txtEndConversationScript=$lower_frame->LabEntry(-textvariable=>\$txtEndConversationScript_var,-background=>'white',
                                           -label=>'Script that fires when conversation ends',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtEndConversationScript->grid('-',-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtEndConversationScript;


    #Abort Conversation Script
    my $txtAbortConversationScript_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EndConverAbort')]{Value};
    my $txtAbortConversationScript=$lower_frame->LabEntry(-textvariable=>\$txtAbortConversationScript_var,-background=>'white',
                                           -label=>'Script that fires if conversation aborts',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtAbortConversationScript->grid('-',-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtAbortConversationScript;

    #VO_ID
    my $txtVO_ID_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('VO_ID')]{Value};
    my $txtVO_ID=$lower_frame->LabEntry(-textvariable=>\$txtVO_ID_var,-background=>'white',
                                           -label=>'Voiceover ID',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtVO_ID->grid('-',-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtVO_ID;

    #AmbientTrack
    my $txtAmbientTrack_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('AmbientTrack')]{Value};
    my $txtAmbientTrack=$lower_frame->LabEntry(-textvariable=>\$txtAmbientTrack_var,-background=>'white',
                                           -label=>'Ambient Track',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtAmbientTrack->grid('-',-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtAmbientTrack;

    #CameraModel
    my $txtCameraModel_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('CameraModel')]{Value};
    my $txtCameraModel=$lower_frame->LabEntry(-textvariable=>\$txtCameraModel_var,-background=>'white',
                                           -label=>'Camera Model',-labelPack=>[-side=>'left',-anchor=>'w'],
                                           -width=>16);
    $txtCameraModel->grid('-',-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtCameraModel;

    #Conversation Type
    my $fraConvType=$lower_frame->Frame(-borderwidth=>2,-relief=>'groove');
    my $optConvType_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ConversationType')]{Value};
    my $optConvTypeNormal=$fraConvType->Radiobutton(-text=>'Human',-value=>0,-variable=>\$optConvType_var);
    my $optConvTypeComputer=$fraConvType->Radiobutton(-text=>'Computer',-value=>1,-variable=>\$optConvType_var);
    my $lblConvType=$fraConvType->Label(-text=>"Conversation Type")->grid('-',-sticky=>'nw');
    $optConvTypeNormal->grid($optConvTypeComputer);
    $fraConvType->grid(-row=>$colrow[0],-column=>0,-sticky=>'ne');
    #$colrow[0]++;
    push @spawned_widgets,($fraConvType,$optConvTypeNormal,$optConvTypeComputer,$lblConvType);

    #Computer Type
    my $fraCompType=$lower_frame->Frame(-borderwidth=>2,-relief=>'groove');
    my $optCompType_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ComputerType')]{Value};
    my $optCompTypeNew=$fraCompType->Radiobutton(-text=>'Modern',-value=>0,-variable=>\$optCompType_var);
    my $optCompTypeOld=$fraCompType->Radiobutton(-text=>'Ancient',-value=>1,-variable=>\$optCompType_var);
    my $lblCompType=$fraCompType->Label(-text=>"Computer Type")->grid('-',-sticky=>'nw');
    $optCompTypeNew->grid($optCompTypeOld,-sticky=>'w');
    $fraCompType->grid(-row=>$colrow[0],-column=>1,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,($fraCompType,$optCompTypeNew,$optCompTypeOld,$lblCompType);

    #Delay Entry
    my $txtDelayEntry_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('DelayEntry')]{Value};
    my $txtDelayEntry=$lower_frame->LabEntry(-textvariable=>\$txtDelayEntry_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'Delay before Entry is spoken',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=0) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid('-',-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtDelayEntry;

    #Delay Reply
    my $txtDelayReply_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('DelayReply')]{Value};
    my $txtDelayReply=$lower_frame->LabEntry(-textvariable=>\$txtDelayReply_var,
                                      -background=>'white',
                                      -width=>6,
                                      -label=>'Delay before Reply is spoken',
                                      -labelPack=>[-side=>'left',-anchor=>'w'],
                                      -validate=>'key',
                                      -validatecommand=>sub{
                                        if ($_[0]=~/^(\d*)$/) {
                                            #valid numeric input
                                            if ($_[0]>=0) {
                                                return 1;
                                            }
                                        }
                                        return 0;
                                      },
                                      -invalidcommand=>sub{$mw->bell})->grid('-',-row=>$colrow[0],-column=>0,-sticky=>'ne');
    $colrow[0]++;
    push @spawned_widgets,$txtDelayReply;

    #checkboxes....
    my $fraCheckboxes=$lower_frame->Frame(-borderwidth=>2,-relief=>'groove');
    push @spawned_widgets,$fraCheckboxes;

    #Skippable
    my $chkSkippable_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('Skippable')]{Value};
    $chkSkippable_var=($chkSkippable_var>0);
    my $chkSkippable=$fraCheckboxes->Checkbutton(-variable=>\$chkSkippable_var,-text=>'Skippable'
                                              )->grid (-sticky=>'nw');
    push @spawned_widgets, $chkSkippable;

    #AnimatedCut
    my $chkAnimatedCut_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('AnimatedCut')]{Value};
    $chkAnimatedCut_var=($chkAnimatedCut_var>0);
    my $chkAnimatedCut=$fraCheckboxes->Checkbutton(-variable=>\$chkAnimatedCut_var,-text=>'AnimatedCut'
                                              )->grid(-sticky=>'nw');
    push @spawned_widgets, $chkAnimatedCut;

    #OldHitCheck
    my $chkOldHitCheck_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('OldHitCheck')]{Value};
    $chkOldHitCheck_var=($chkOldHitCheck_var>0);
    my $chkOldHitCheck=$fraCheckboxes->Checkbutton(-variable=>\$chkOldHitCheck_var,-text=>'OldHitCheck'
                                              )->grid(-sticky=>'nw');

    push @spawned_widgets, $chkOldHitCheck;

    #UnequipHItem
    my $chkUnequipHItem_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('UnequipHItem')]{Value};
    $chkUnequipHItem_var=($chkUnequipHItem_var>0);
    my $chkUnequipHItem=$fraCheckboxes->Checkbutton(-variable=>\$chkUnequipHItem_var,-text=>'UnequipHItem'
                                              )->grid(-sticky=>'nw');

    push @spawned_widgets, $chkUnequipHItem;

    #UnequipItems
    my $chkUnequipItems_var=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('UnequipItems')]{Value};
    $chkUnequipItems_var=($chkUnequipItems_var>0);
    my $chkUnequipItems=$fraCheckboxes->Checkbutton(-variable=>\$chkUnequipItems_var,-text=>'UnequipItems'
                                              )->grid(-sticky=>'nw');
    push @spawned_widgets, $chkUnequipItems;
    $fraCheckboxes->grid(-rowspan=>5, -row=>1, -column=>3);

    #Stunt List
    my $stuntlist_frame=$lower_frame->Frame()->grid(-row=>1,-column=>4,-rowspan=>5);
    my $lblStuntList=$stuntlist_frame->Label(-text=>'Cutscene Model [Participant]')->grid;

    my $stuntlist=$stuntlist_frame->Scrolled("Listbox",-scrollbars=>'oe',
                                                      -height=>'4',-width=>'30',
                                                      -background=>'white');
    $stuntlist->bind('<Button-3>'=>sub { my $stuntlist_item=shift; stuntlist_rclick($stuntlist_item) });




    $stuntlist->grid;

    #$mw->bind('<KeyPress>'=>\&print_keysym); #for binding debugging only
    $mw->bind('<KeyRelease-Delete>'=>sub {
        if ($stuntlist->curselection) { $stuntlist->delete($stuntlist->curselection)}
    });

    my $stuntlist_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StuntList')]{Value};
    if (ref $stuntlist_arr_ref eq 'ARRAY') {
        for my $stunt_struct (@$stuntlist_arr_ref) {
            my $this_stuntmodel=$stunt_struct->{Fields}[$stunt_struct->get_field_ix_by_label('StuntModel')]{Value};
            my $this_participant=$stunt_struct->{Fields}[$stunt_struct->get_field_ix_by_label('Participant')]{Value};
            $stuntlist->insert('end',"$this_stuntmodel [$this_participant]");
        }
    }
    push @spawned_widgets, ($stuntlist_frame,$lblStuntList,$stuntlist);

    #Apply Changes
    my $btnApply=$lower_frame->Button(-text=>'Apply Changes',
                                      -borderwidth=>2,
                                      -width=>25,
                                      -height=>2,
                                      -command=>sub {
        # Update the GFF object
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EndConversation')]{Value}=$txtEndConversationScript_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EndConverAbort')]{Value}=$txtAbortConversationScript_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('VO_ID')]{Value}=$txtVO_ID_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('AmbientTrack')]{Value}=$txtAmbientTrack_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('CameraModel')]{Value}=$txtCameraModel_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ConversationType')]{Value}=$optConvType_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ComputerType')]{Value}=$optCompType_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('DelayEntry')]{Value}=$txtDelayEntry_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('DelayReply')]{Value}=$txtDelayReply_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('Skippable')]{Value}=$chkSkippable_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('AnimatedCut')]{Value}=$chkAnimatedCut_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('UnequipHItem')]{Value}=$chkUnequipHItem_var;
        $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('UnequipItems')]{Value}=$chkUnequipItems_var;

        my @gff_stunt_array=();
        for my $stuntitem ($stuntlist->get(0,'end')) {
            next unless $stuntitem=~/(\w+) \[(\w+)\]/;
            my $new_model=$1;
            my $new_participant=$2;
            my $new_stunt=Bioware::GFF::Struct->new('ID'=>0);
            $new_stunt->createField('Type'=>FIELD_CEXOSTRING,'Label'=>'Participant','Value'=>$new_participant);
            $new_stunt->createField('Type'=>FIELD_RESREF,'Label'=>'StuntModel','Value'=>$new_model);
            push @gff_stunt_array, $new_stunt;
        }
        if (defined $gff->{Main}->get_field_ix_by_label('StuntList')) {
         $gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StuntList')]{Value}=[@gff_stunt_array];
        }
        else {
         $gff->{Main}->createField('Type'=>FIELD_LIST,'Label'=>'StuntList','Value'=>[@gff_stunt_array]);
        }
        gff_updated();
                                        })->grid(-column=>3,-row=>7,-rowspan=>2,-columnspan=>2);
    push @spawned_widgets,$btnApply;
}
sub get_parent_child_link_structs {

    # Inputs:
    #  a treeitem (scalar)
    # Returns:
    #  S, E, or R depending on if the parent is a startinglist, entry, or reply struct (scalar)
    #  parent's Entry or Reply (Bioware::GFF::Struct) (empty if StartingList)
    #  index of said struct (scalar)
    #  parent's RepliesList, EntriesList, or StartingList (Bioware::GFF::Struct) that contains the link to the child
    #  index of said substruct (scalar)
    #  ref to array of substructs

    # called by cut_node() and show_detail_widgets()

    my $treeitem=shift;
    #who's your daddy?
    my $parent=$tree->info('parent',$treeitem);
    my $parent_arr_ref;
    my $parent_type;
    my $parent_lastbranch;
    my $parent_struct_index;
    my $parent_struct;
    my $parent_substruct_index;
    my $parent_substruct_arr_ref;
    my $parent_substruct;
    if ($treeitem =~/O/) { #no parent, child is orphan, how sad
        $parent_type='O';
    }
    elsif ($parent eq '#') {
        #parent is StartingList
        $parent_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('StartingList')]{Value};
        $parent_type='S';
        my @parent_children=$tree->info('children',$parent);
        my $i=0;
        for my $parent_child (@parent_children) {
            next if $parent_child =~ /O/;                   #ignore orphans
            if ($parent_child eq $treeitem) {
                $parent_substruct_index=$i;
                $parent_substruct_arr_ref=$parent_arr_ref;  # simple alias...
                $parent_substruct=$$parent_arr_ref[$parent_substruct_index];
                last;
            }
            $i++;
        }
    }
    else {
        $parent_lastbranch=(split /#/,$parent)[-1];
        $parent_lastbranch=~/(\d+)/;
        $parent_struct_index=$1;
        if ($parent_lastbranch =~ /E/) {
            #parent is Entry
            $parent_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('EntryList')]{Value};
            $parent_type='E';
            $parent_struct=$$parent_arr_ref[$parent_struct_index];
            my @parent_children=$tree->info('children',$parent);
            for (my $i=0; $i<scalar @parent_children; $i++) {
                if ($parent_children[$i] eq $treeitem) {
                    $parent_substruct_index=$i;
                    last;
                }
            }
            $parent_substruct_arr_ref=$parent_struct->{Fields}[$parent_struct->get_field_ix_by_label('RepliesList')]{Value};
            $parent_substruct=$$parent_substruct_arr_ref[$parent_substruct_index];
        }
        elsif ($parent_lastbranch =~ /R/) {
            #parent is Reply
            $parent_arr_ref=$gff->{Main}{Fields}[$gff->{Main}->get_field_ix_by_label('ReplyList')]{Value};
            $parent_type='R';
            $parent_struct=$$parent_arr_ref[$parent_struct_index];
            my @parent_children=$tree->info('children',$parent);
            for (my $i=0; $i<scalar @parent_children; $i++) {
                if ($parent_children[$i] eq $treeitem) {
                    $parent_substruct_index=$i;
                    last;
                }
            }
            $parent_substruct_arr_ref=$parent_struct->{Fields}[$parent_struct->get_field_ix_by_label('EntriesList')]{Value};
            $parent_substruct=$$parent_substruct_arr_ref[$parent_substruct_index];
        }

    }
    return ($parent_type, $parent_struct, $parent_struct_index, $parent_substruct, $parent_substruct_index, $parent_substruct_arr_ref);
}

sub tree_dblclick {

    my $treeitem=shift;
    # now perform right-click actions...
    return unless ($treeitem);
    return if $treeitem eq '#';
    return unless $tree->infoExists($treeitem);
    my $text=$tree->entrycget($treeitem,-text);
    return unless $text=~/(already listed)/;
    jump_to_original($treeitem);


}
sub jump_to_original {
    # this function will find the original instance of the Reply/Entry in the tree
    # and open up the leaf so that the node is exposed
    # and will select the node and make sure it is visible on the user's screen    my $lastbranch=(split /#/,$treeitem)[-1];
    my $treeitem=shift;
    my $lastbranch=(split /#/,$treeitem)[-1];
    $lastbranch=~/(\d+)/;
    my $index=$1;
    return unless ($lastbranch);

    # first find the original
    my $ti='#';
    while ($ti=$tree->infoNext($ti)) {          # the first one found is the original
        if ($ti =~ /$lastbranch$/) {
            last;
        }
    }
    return unless ($ti);

    # now open up leaf to this node
    my @nodes = split /#/,$ti;
    my $thisnode;
    for (my $i=1; $i<scalar @nodes; $i++) {     # start at 1, skipping the first element that represents the root node
        $thisnode=join '#', @nodes[0..$i];
        $tree->show('entry',$thisnode);
    }

    #now select node
    $tree->selectionClear();
    $tree->selectionSet($ti);

    #now see the node
    $tree->see($ti);

    #spawn widgets ala left click
    tree_click($ti);

}

sub stuntlist_rclick {
   my $w=shift;
   my $e = $w->XEvent();
   my $idx = $w->nearest($e->y); # get the listbox entry under cursor
   if ($idx>=0) {
      $w->selectionSet($idx);
   }
   #now perform right-click actions...

   my $context_menu = $mw->Menu(-tearoff  => 0);
   $context_menu->add('command',-label=>'Add StuntList Item',-command=>sub{
      my $d = $mw->DialogBox(-title => "Enter Stunt Model resref",-buttons => ["OK","Cancel"]);
      my $new_stuntmodel;
      $d->add('Label', -text=>"Enter the Stunt Model resource reference")->pack;
      $d->add('Entry', -textvariable=>\$new_stuntmodel, -width=>16)->pack;
      my $answer=$d->Show(-popover  => 'cursor',-popanchor => 'nw');
      return if $answer eq 'Cancel';
      $d = $mw->DialogBox(-title => "Enter Participant",-buttons => ["OK","Cancel"]);
      my $new_participant='PLAYER';
      $d->add('Label', -text=>"Enter the Participant for Stunt")->pack;
      $d->add('Entry', -textvariable=>\$new_participant, -width=>16)->pack;
      $answer=$d->Show(-popover  => 'cursor',-popanchor => 'sw');
      return if $answer eq 'Cancel';
      $w->insert('end',"$new_stuntmodel [$new_participant]");
      gff_updated();
   });
   if ($idx>=0) {
      $context_menu->add('command',-label=>'Modify This Stunt',-command=>sub{
         my $item=$w->get($idx);
         return unless $item=~/(\w+) \[(\w+)\]/;
         my $new_stuntmodel=$1;
         my $new_participant=$2;
         my $d = $mw->DialogBox(-title => "Enter Stunt Model resref",-buttons => ["OK","Cancel"]);
         $d->add('Label', -text=>"Enter the Stunt Model resource reference")->pack;
         $d->add('Entry', -textvariable=>\$new_stuntmodel, -width=>16)->pack;
         my $answer=$d->Show(-popover  => 'cursor',-popanchor => 'nw');
         return if $answer eq 'Cancel';
         $d = $mw->DialogBox(-title => "Enter Participant",-buttons => ["OK","Cancel"]);
         $d->add('Label', -text=>"Enter the Participant for Stunt")->pack;
         $d->add('Entry', -textvariable=>\$new_participant, -width=>16)->pack;
         $answer=$d->Show(-popover  => 'cursor',-popanchor => 'sw');
         return if $answer eq 'Cancel';
         $w->insert($idx,"$new_stuntmodel [$new_participant]");
         $w->delete($idx+1);
         gff_updated();
      });
      $context_menu->add('command',-label=>'Delete This Stunt',-command=>sub{
         $w->delete($idx); gff_updated(); });
    }
   $context_menu->Popup(
    -popover  => 'cursor',
    -popanchor => 'nw');
}
sub animation_list_rclick {
   my $w=shift;
   my $w_src=shift;
   my $e = $w->XEvent();
   my $idx = $w->nearest($e->y); # get the listbox entry under cursor
   if ($idx>=0) {$w->selectionSet($idx)}

   #now perform right-click actions...
   my $context_menu = $mw->Menu(-tearoff  => 0);

   $context_menu->add('command',-label=>'Add New Animation',-command=>sub{
      my $new_animation;
      my $new_participant='OWNER';
      my $d = $mw->DialogBox(-title => "Select Animation",-buttons => ["OK","Cancel"]);
      $d->add('Label', -text=>"Select the Animation")->pack;
      $d->add('BrowseEntry', -colorstate=>'white',-choices=>[($w_src->get('0','end'))], -state=>'readonly',-variable=>\$new_animation, -width=>25)->pack;
      my $answer=$d->Show(-popover  => 'cursor',-popanchor => 'nw');
      return if $answer eq 'Cancel';
      $d = $mw->DialogBox(-title => "Enter Participant",-buttons => ["OK","Cancel"]);
      $d->add('Label', -text=>"Enter the Participant for Animation")->pack;
      $d->add('Entry', -textvariable=>\$new_participant, -width=>16)->pack;
      $answer=$d->Show(-popover  => 'cursor',-popanchor => 'sw');
      return if $answer eq 'Cancel';
      $w->insert($idx,"$new_animation [$new_participant]");
      $w->delete($idx+1);
      gff_updated();
   });
   if ($idx>=0) {
      $context_menu->add('command',-label=>'Modify This Animation',-command=>sub{
         my $item=$w->get($idx);
         return unless $item=~/(\w+) \[(\w+)\]/;
         my $new_animation=$1;
         my $new_participant=$2;
         my $d = $mw->DialogBox(-title => "Select Animation",-buttons => ["OK","Cancel"]);
         $d->add('Label', -text=>"Select the Animation")->pack;
         $d->add('BrowseEntry', -choices=>[($w_src->get('0','end'))], -state=>'readonly',-variable=>\$new_animation, -width=>25)->pack;
         my $answer=$d->Show(-popover  => 'cursor',-popanchor => 'nw');
         return if $answer eq 'Cancel';
         $d = $mw->DialogBox(-title => "Enter Participant",-buttons => ["OK","Cancel"]);
         $d->add('Label', -text=>"Enter the Participant for Animation")->pack;
         $d->add('Entry', -textvariable=>\$new_participant, -width=>16)->pack;
         $answer=$d->Show(-popover  => 'cursor',-popanchor => 'sw');
         return if $answer eq 'Cancel';
         $w->insert($idx,"$new_animation [$new_participant]");
         $w->delete($idx+1);
         gff_updated();
      });

      $context_menu->add('command',-label=>'Delete This Animation',-command=>sub{
         $w->delete($idx); gff_updated(); });
   }
   $context_menu->Popup(
    -popover  => 'cursor',
    -popanchor => 'nw');


}
sub change_dialog_dot_tlk {
   my $new_path=OpenDialog(
                           title=>'Locate Dialog.tlk file',
                           filters=>['TLK file'=>'*.tlk'],
                           options=>OFN_FILEMUSTEXIST|OFN_HIDEREADONLY);
   if ($new_path) {
      $new_path=~/(.*)\\/;
      $path_to_dialog_dot_tlk=$1;
   }

}
sub launch_dlgedit {
   my $parm;
   if ($current_dialog){
      $parm='& Chr(34) & "'.$current_dialog.'" & Chr(34)';
   }
   if (-e 'DLGEdit.exe') {
      open H,">dlged.vbs";
      print H 'set WshShell = WScript.CreateObject("WScript.Shell")'."\n";
      print H 'WshShell.Run "dlgedit.exe "'.$parm.',1';
      close H;
      system 'dlged.vbs';
      }
   else {
      #my $d = $mw->DialogBox(-title => "DLGEdit Not Found",-buttons => ["OK"]);
      #$d->add('Label', -text=>'DLGEdit was not found in this directory')->pack;
      #$d->Show();
      my $new_path=OpenDialog(
                           title=>'Locate DLGEdit.exe file',
                           filters=>['DLGEdit.exe'=>'DLGEdit.exe'],
                           options=>OFN_FILEMUSTEXIST|OFN_HIDEREADONLY);
      if ($new_path) {
      open H,">dlged.vbs";
      print H 'set WshShell = WScript.CreateObject("WScript.Shell")'."\n";
      print H 'WshShell.Run "'.$new_path.' "'. $parm.',1';
      close H;
      system 'dlged.vbs';
      }
   }
}
sub launch_gffeditor {
   if (-e 'GFFEditor.exe') { system "GFFEditor.exe \"$current_dialog\"" }
   else {
      #my $d = $mw->DialogBox(-title => "GFFEditor Not Found",-buttons => ["OK"]);
      #$d->add('Label', -text=>'GFF Editor was not found in this directory')->pack;
      #$d->Show();
      my $new_path=OpenDialog(
                           title=>'Locate GFFEditor.exe file',
                           filters=>['GFFEditor.exe'=>'GFFEditor.exe'],
                           options=>OFN_FILEMUSTEXIST|OFN_HIDEREADONLY);
      if ($new_path) { system "\"$new_path\" \"$current_dialog\"" }
   }
}

sub delete_startinglist_structs_by_index {
   my $ix_to_find=shift;
   return unless $gff;
   my $startinglist_ix=$gff->{Main}->get_field_ix_by_label('StartingList');
   my $startinglist_arr_ref=$gff->{Main}{Fields}[$startinglist_ix]{Value};
   my @new_arr;
   for my $startinglist_struct (@$startinglist_arr_ref) {
      my $index_ix=$startinglist_struct->get_field_ix_by_label('Index');
      unless($startinglist_struct->{Fields}[$index_ix]{Value}==$ix_to_find) {
         $startinglist_struct->{ID}=scalar @new_arr;
         push @new_arr,$startinglist_struct;
      }
   }
   $gff->{Main}{Fields}[$startinglist_ix]{Value}=[@new_arr];
}
sub delete_replieslist_structs_by_index {
   my $ix_to_find=shift;
   return unless $gff;
   my $entrylist_ix=$gff->{Main}->get_field_ix_by_label('EntryList');
   my $entrylist_arr_ref=$gff->{Main}{Fields}[$entrylist_ix]{Value};
   for (my $entry_struct_ix=0; $entry_struct_ix<scalar @$entrylist_arr_ref; $entry_struct_ix++) {
      my $entry_struct=$gff->{Main}{Fields}[$entrylist_ix]{Value}[$entry_struct_ix];
      my $replieslist_ix=$entry_struct->get_field_ix_by_label('RepliesList');
      if (defined $replieslist_ix) {
         my $replieslist_arr_ref=$entry_struct->{Fields}[$replieslist_ix]{Value};
         my @new_arr;
         for my $replieslist_struct (@$replieslist_arr_ref) {
            my $index_ix=$replieslist_struct->get_field_ix_by_label('Index');
            if (defined $index_ix) {
               unless ($ix_to_find==$replieslist_struct->{Fields}[$index_ix]{Value}) {
                  $replieslist_struct->{ID}=scalar @new_arr;
                  push @new_arr, $replieslist_struct;
               }
            }
         }
         $entry_struct->{Fields}[$replieslist_ix]{Value}=[@new_arr];
      }
      $gff->{Main}{Fields}[$entrylist_ix]{Value}[$entry_struct_ix]=$entry_struct;
   }
}
sub delete_entrieslist_structs_by_index {
   my $ix_to_find=shift;
   return unless $gff;
   my $replylist_ix=$gff->{Main}->get_field_ix_by_label('ReplyList');
   my $replylist_arr_ref=$gff->{Main}{Fields}[$replylist_ix]{Value};
   for (my $reply_struct_ix=0; $reply_struct_ix<scalar @$replylist_arr_ref; $reply_struct_ix++) {
      my $reply_struct=$gff->{Main}{Fields}[$replylist_ix]{Value}[$reply_struct_ix];
      my $entrieslist_ix=$reply_struct->get_field_ix_by_label('EntriesList');
      if (defined $entrieslist_ix) {
         my $entrieslist_arr_ref=$reply_struct->{Fields}[$entrieslist_ix]{Value};
         my @new_arr;
         for my $entrieslist_struct (@$entrieslist_arr_ref) {
            my $index_ix=$entrieslist_struct->get_field_ix_by_label('Index');
            if (defined $index_ix) {
               unless ($ix_to_find==$entrieslist_struct->{Fields}[$index_ix]{Value}) {
                  $entrieslist_struct->{ID}=scalar @new_arr;
                  push @new_arr, $entrieslist_struct;
               }
            }
         }
         $reply_struct->{Fields}[$entrieslist_ix]{Value}=[@new_arr];
      }
      $gff->{Main}{Fields}[$replylist_ix]{Value}[$reply_struct_ix]=$reply_struct;
   }
}
sub Tk::Error   #custom error handler --> error log, and error msg
{
    my $w = shift;
    my $error = shift;
    if (Exists($w))
    {
         my $grab = $w->grab('current');
        $grab->Unbusy if (defined $grab);
    }
    chomp($error);
    my $errmessages="$error\n " . join("\n ",@_);
    my $d = $mw->DialogBox(-title => "Err Message",-buttons => ["OK"]);
    $d->add('Label', -text=>$errmessages)->pack;
    my $answer=$d->Show();
}
